package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.forest.ForestSource;
import com.almworks.jira.structure.api.item.ItemTracker;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.atlassian.annotations.Internal;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import java.util.Collection;

/**
 * <p>{@code ValuesMeta} contains additional information about an attribute loading process, executed by calling
 * {@link StructureAttributeService#loadAttributeValues}. An instance of {@code ValuesMeta} is received via
 * {@link AttributeValuesReceiver#receiveMeta(ValuesMeta)} method.</p>
 *
 * <p>{@code ValuesMeta} may be used to extract diagnostic information about failed attributes. It can also be used to check if the
 * there may be any updates to the values loaded.</p>
 *
 * @see StructureAttributeService#hasUpdate
 * @see #getLoadingErrors()
 */
@PublicApi
public interface ValuesMeta {
  /**
   * Returns the errors encountered during loading the values.
   *
   * @return a collection of loading errors
   */
  @NotNull
  Collection<AttributeErrorInfo> getLoadingErrors();

  /**
   * Returns the version of the item change stream at a moment before all the values were loaded. Any subsequent item changes might cause
   * some of the loaded values to be recalculated.
   *
   * @return data version of the item change stream
   * @see ItemTracker
   */
  @NotNull
  DataVersion getItemsVersion();

  /**
   * Returns the version of the forest that was used to load the values. Returns {@link DataVersion#ZERO} if the
   * forest was non-versioned (based on {@code ItemForest}).
   *
   * @return version of the forest
   * @see ForestSource
   */
  @NotNull
  DataVersion getForestVersion();

  /**
   * Returns the version of the item-based value cache at which the values were loaded. This version reflects the
   * sequence of invalidation events in the cache. This method is intended for internal use.
   *
   * @return version of the item-based value cache
   */
  @Internal
  @NotNull
  DataVersion getItemCacheSequence();

  /**
   * Returns the version of the forest-based value cache at which the values were loaded, or {@link DataVersion#ZERO} if
   * the forest was not versioned. This version reflects the sequence of invalidation events in the cache.
   * This method is intended for internal use.
   *
   * @return version of the forest-based value cache
   */
  @Internal
  @NotNull
  DataVersion getForestCacheSequence();
}
