package com.almworks.jira.structure.api.attribute;

import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>Represents a value loaded from the attribute system. Contains the {@link AttributeValue} and a boolean {@code outdated} flag.</p>
 *
 * <p>Note that the object is immutable with regards to the attribute value only. Outdated flag may change from {@code false} to {@code true}
 * once during the lifespan of a loaded value.</p>
 *
 * @param <T> value type
 * 
 * @see LoadedValues
 * @see StructureAttributeService#getAttributeValues
 */
@PublicApi
public interface LoadedValue<T> {
  /**
   * Returns the loaded attribute value.
   *
   * @return attribute value
   */
  @NotNull
  AttributeValue<T> getValue();

  /**
   * <p>Returns {@code true} if this value has been outdated. An outdated value will be recalculated next time it is requested.</p>
   *
   * <p>A value may be outdated right away, or it may become outdated later.</p>
   *
   * @return true if the value is outdated
   */
  boolean isOutdated();

  /**
   * Creates an instance of {@link LoadedValue} based on attribute value. This loaded value will never be outdated.
   *
   * @param attributeValue attribute value
   * @param <T> value type
   * @return loaded value
   */
  static <T> LoadedValue<T> of(@NotNull AttributeValue<T> attributeValue) {
    //noinspection ConstantConditions
    if (attributeValue == null) {
      throw new NullPointerException();
    }
    return new UpToDateLoadedValue<>(attributeValue);
  }

  /**
   * Creates an instance of {@link LoadedValue} based on a value itself. This loaded value will never be outdated.
   *
   * @param value value
   * @param <T> value type
   * @return loaded value
   */
  static <T> LoadedValue<T> of(@Nullable T value) {
    return new UpToDateLoadedValue<>(AttributeValue.ofNullable(value));
  }
}
