package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.row.StructureRow;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>Scanning attributes are used to process and aggregate values across all the forest, from top to bottom (depth-first traversal).
 * Imagine fully expanding the forest and then processing each row, starting with the top and going down, aggregating a resulting value.
 * For example, a rolling total can be implemented with a scanning attribute - it will show the total of some value of all the rows above
 * the current one.</p>
 *
 * <p>Scanning loader represents a scanning attribute. Scanning loading function for a row {@code R}
 * can calculate value based on:</p>
 * <ul>
 * <li>Already calculated value of the same attribute for the row, immediately preceding R;</li>
 * <li>Values of attribute dependencies for the row R;</li>
 * <li>{@link StructureRow} data for {@code R} and its immediate predecessor;</li>
 * <li>Forest depth of {@code R} and its immediate predecessor;</li>
 * <li>Context values, if declared with {@link #getContextDependencies}.</li>
 * </ul>
 *
 * <p>The immediate predecessor is calculated without regarding the hierarchy. If {@code R} has index {@code i} in the forest, its immediate
 * predecessor will have index {@code i - 1}. See {@link Forest#getRow(int)}.</p>
 *
 * <p><strong>The loader must be a stable function.</strong> It must not depend on any other inputs, and should
 * provide the same result for the same inputs every time: {@code V(row) = SCANNING(V(predecessor(row)), dependencies(row))}.</p>
 *
 * <p>Similar to aggregates and propagates, this loader has a recursive definition and its properties can be derived by closing the recursion.</p>
 *
 * <h3>Using loader data</h3>
 *
 * <p>Note that sometimes a loader will need to store more data in {@link AttributeValue} than the resulting Java
 * type allows. In that case, use loader data - see {@link AttributeValue#withData}.</p>
 *
 * <h3>Sensitive data control</h3>
 *
 * <p>A scanning attribute, as well as all multi-row loaders, is potentially capable of accessing data calculated for
 * rows that the requesting user is not allowed to see. For example, an allocated budget may be inherited from a larger-scope budgeting
 * parent issue, and a user may not have access to that issue.</p>
 *
 * <p>If the attribute collected is
 * considered sensitive (according to {@link com.almworks.jira.structure.api.settings.StructureConfiguration#getAttributeSensitivitySettings()}), then
 * the calls to {@link AttributeLoaderContext#getDependencyValue} will not disclose the hidden value. Instead,
 * the loader function will receive {@link AttributeValue#inaccessible()}.</p>
 *
 * <p>In other words, you should not worry about implementing security checks in a scanning attribute, the system does
 * it for you.</p>
 *
 * <h3>No super-root support</h3>
 *
 * <p>Scanning attribute loaders do not support super-root and will return undefined value for it all the time. When calculating {@code 0}th row's
 * value, the preceding value will be {@link AttributeValue#undefined()}, and {@link ScanningAttributeContext#getPrecedingRow()} will return
 * {@code null}.</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 * @see AggregateAttributeLoader
 * @see PropagateAttributeLoader
 * @see com.almworks.jira.structure.api.settings.AttributeSensitivitySettings
 */
@PublicSpi
public interface ScanningAttributeLoader<T> extends MultiRowAttributeLoader<T> {
  /**
   * <p>Performs scanning loading. Must be a pure function - see {@link ScanningAttributeLoader}.</p>
   *
   * <p>If the value is missing or empty, the loader should return {@link AttributeValue#undefined()}. If for some reason this loader
   * is not applicable for the given row, it should return {@code null}. (Note, however, that if multiple scanning loaders work on the same
   * attribute, it could be tricky and lead to unexpected results.)</p>
   *
   * @param precedingValue the value of this attribute for the immediately preceding row, or {@link AttributeValue#undefined()} for the very first row
   * @param context loading context
   * @return the calculated value
   */
  @Nullable
  AttributeValue<T> loadValue(@NotNull AttributeValue<T> precedingValue, @NotNull ScanningAttributeContext context);
}
