package com.almworks.jira.structure.api.backup;

import com.almworks.jira.structure.api.process.ProgressGauge;
import org.jetbrains.annotations.NotNull;

import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * <p>Restore operation reads a backup file and loads it into the Structure database.</p>
 *
 * <p>Note that if there's existing data in the structure database, it's not removed, but rather
 * written over. If there's a structure with an ID that is present in the backup file, it will be
 * overwritten. If there's a structure that's not present in the backup file, it will remain.</p>
 */
public interface RestoreOperation {
  /**
   * Sets the location of the backup file.
   *
   * @param file backup file
   * @return this operation
   */
  @NotNull
  RestoreOperation setFile(@NotNull File file);

  /**
   * <p>Performs the restore from backup synchronously. After this method has finished successfully, the structure
   * data contains all structures and other data from the backup file.</p>
   *
   * <p>Issues in the backup file are referred to by their ID.</p>
   *
   * <p>During this operation all exceptions or not found issues in JIRA are collected in {@link #getErrorsContainer()}</p>
   *
   * @return this operation
   * @throws IOException if the file cannot be read or has invalid format
   */
  @NotNull
  RestoreOperation restore() throws IOException;

  /**
   * Checks that backup file has correct format. Returns normally if the file can be used to restore data
   * and throws exception if not.
   *
   * @return this operation
   * @throws IOException if the file cannot be read or has invalid format
   */
  @NotNull
  RestoreOperation checkBackupFile() throws IOException;

  /**
   * Call this method after {@link #restore()} has completed to get names of all restored structures.
   *
   * @return map of structure names. Key is the structure ID, and value is the structure name.
   */
  @NotNull
  Map<Long, String> getStructureNames();

  /**
   * @return container with invalid issues (not found in JIRA), synchronizers that wasn't correctly processed during restore
   */
  ErrorsContainer getErrorsContainer();

  /**
   * @return a {@link ProgressGauge} allowing you to check this operation's progress and cancel it in a different thread.
   */
  ProgressGauge getProgressGauge();

  /**
   * Sets the <code>restoreHistory</code> flag. When the flag is set, the histories will be restored.
   *
   * @param restoreHistory default value is <b>true</b>
   * @return this operation
   */
  @NotNull
  RestoreOperation setRestoreHistory(boolean restoreHistory);
}
