package com.almworks.jira.structure.api.attribute;

import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * <p>Provides normalization for attribute spec parameters.</p>
 *
 * <p>Attribute spec parameters are normalized. This means that if the value of a parameter is the default for that
 * parameter, it is not stored in the map. For example, {@code int} parameters have default equal to {@code 0}. If you try
 * to build an attribute spec and set an {@code int} parameter to 0, the resulting attribute spec will not have that
 * value.</p>
 *
 * <p>This is done to avoid unequal instances of {@code AttributeSpec} to represent the same attribute. If not normalized,
 * multiple versions of the same attribute spec could lead to bugs.</p>
 *
 * <h3>Non-standard defaults</h3>
 *
 * <p>Some parameters for some specs are known to have non-standard defaults. The normalization then works by removing the parameter
 * only if it is set to the default value.</p>
 *
 * <p>For example, for {@code "aggregation-parent"} attribute if you set the {@code "level"} parameter to {@code -1}, it will be removed, but if you
 * set it to {@code 0}, it will not be removed.</p>
 *
 * <p>See {@link #KNOWN_NON_STANDARD_INT_DEFAULTS}.</p>
 *
 * <h3>Normalizing attribute parameters</h3>
 *
 * <p>If there's an object parameter that looks like it's an attribute parameter (has {@code "id"} parameter itself), then it will also
 * be normalized.</p>
 */
@Internal
public class AttributeSpecNormalization {
  /**
   * By default, all int parameters are supposed to default to 0. Thus, if an int parameter is set to 0, it is removed from the parameter map.
   * For some known specs, this doesn't work - see AggregationLoaderProvider.AGGREGATION_PARENT. This field lists known exceptions to the rule.
   * (Should also be reflected in AttributeSpecNormalization.js)
   */
  private static final Map<String, Map<String, Integer>> KNOWN_NON_STANDARD_INT_DEFAULTS = immutableMapOf(
    "aggregation-parent", immutableMapOf("depth", -1, "level", -1),
    "aggregation-join", immutableMapOf("fromdepth", 1, "fromLevel", 1)
  );

  /**
   * Does normalization based on the rules at https://dev.almworks.com/browse/STR-1464
   * See also AttributeSpecNormalization.js
   *
   * @param specParams
   * @return normalized specParams
   */
  @NotNull
  public static Map<String, Object> normalizeParams(@NotNull String specId, @NotNull Map<String, Object> specParams) {
    if (!isNormalizationNeededForSpec(specId, specParams)) return specParams;
    return normalizeMap(specId, specParams);
  }

  private static boolean isNormalizationNeededForSpec(String specId, Map<String, Object> params) {
    for (Map.Entry<String, Object> e : params.entrySet()) {
      if (isNormalizationNeededForParameter(specId, e.getKey(), e.getValue())) return true;
    }
    return false;
  }

  private static boolean isNormalizationNeededForParameter(String specId, String paramId, Object value) {
    if (isValueRemoved(value, specId, paramId)) return true;
    return isNormalizationNeededForValue(value);
  }

  private static boolean isNormalizationNeededForValue(Object value) {
    SpecParam spec = SpecParam.fromValue(value);
    if (spec != null) {
      return isNormalizationNeededForSpec(spec.id, spec.params);
    }
    if (value instanceof List) {
      //noinspection unchecked
      return isNormalizationNeededForList((List<Object>) value);
    }
    return false;
  }

  private static boolean isNormalizationNeededForList(List<Object> value) {
    for (Object element : value) {
      if (isNormalizationNeededForValue(element)) return true;
    }
    return false;
  }

  @NotNull
  private static Map<String, Object> normalizeMap(String specId, @NotNull Map<String, Object> specParams) {
    if (specParams.isEmpty()) {
      return specParams;
    }
    LinkedHashMap<String, Object> r = new LinkedHashMap<>(specParams.size());
    for (Map.Entry<String, Object> e : specParams.entrySet()) {
      String paramId = e.getKey();
      Object value = normalizeValue(e.getValue());
      if (!isValueRemoved(value, specId, paramId)) {
        r.put(paramId, value);
      }
    }
    return r;
  }

  private static List<Object> normalizeList(@NotNull List<Object> list) {
    if (list.isEmpty()) {
      return list;
    }
    ArrayList<Object> r = new ArrayList<>(list.size());
    for (Object value : list) {
      r.add(normalizeValue(value));
    }
    return r;
  }

  @Nullable
  @SuppressWarnings("unchecked")
  private static Object normalizeValue(Object value) {
    SpecParam spec = SpecParam.fromValue(value);
    if (spec != null) {
      LinkedHashMap<String, Object> r = new LinkedHashMap<>(3);
      r.put(SharedAttributeSpecs.Param.ID, spec.id);
      if (spec.format != null) r.put(SharedAttributeSpecs.Param.FORMAT, spec.format);
      Map<String, Object> params = normalizeMap(spec.id, spec.params);
      if (!params.isEmpty()) r.put(SharedAttributeSpecs.Param.PARAMS, params);
      return r;
    }
    if (value instanceof List) {
      return normalizeList((List<Object>) value);
    }
    return value;
  }

  private static boolean isValueRemoved(Object value, String specId, String paramId) {
    if (value == null) return true;
    if (value instanceof Boolean && !(boolean) value) return true;
    if (value instanceof Integer && (int) value == getIntDefault(specId, paramId)) return true;
    if (value instanceof Long && (long) value == 0L) return true;
    if (value instanceof List && ((List) value).isEmpty()) return true;
    if (value instanceof Map && ((Map) value).isEmpty()) return true;
    return false;
  }

  private static int getIntDefault(String specId, String paramId) {
    Map<String, Integer> map = KNOWN_NON_STANDARD_INT_DEFAULTS.get(specId);
    if (map != null) {
      Integer defaultValue = map.get(paramId);
      if (defaultValue != null) {
        return defaultValue;
      }
    }
    return 0;
  }


  private static class SpecParam {
    @NotNull
    public final String id;

    @NotNull
    public final Map<String, Object> params;

    @Nullable
    public final String format;

    public SpecParam(@NotNull String id, @NotNull Map<String, Object> params, @Nullable String format) {
      this.id = id;
      this.params = params;
      this.format = format;
    }

    @SuppressWarnings("unchecked")
    @Nullable
    public static SpecParam fromValue(Object value) {
      if (!(value instanceof Map)) return null;
      Map<String, Object> object = (Map<String, Object>) value;
      Object id = null, params = null, format = null;
      for (Map.Entry<String, Object> e : object.entrySet()) {
        String key = e.getKey();
        if (SharedAttributeSpecs.Param.ID.equals(key)) {
          id = e.getValue();
        } else if (SharedAttributeSpecs.Param.PARAMS.equals(key)) {
          params = e.getValue();
        } else if (SharedAttributeSpecs.Param.FORMAT.equals(key)) {
          format = e.getValue();
        } else {
          // not an attribute spec
          return null;
        }
      }
      if (!(id instanceof String) || !(params instanceof Map)) return null;
      String stringId = (String) id;
      if (stringId.isEmpty()) return null;
      if (format != null && !(format instanceof String)) return null;
      return new SpecParam(stringId, (Map<String, Object>) params, (String) format);
    }
  }

  private static <K, V> Map<K, V> immutableMapOf(K key, V value) {
    LinkedHashMap<K, V> r = new LinkedHashMap<>();
    r.put(key, value);
    return Collections.unmodifiableMap(r);
  }

  private static <K, V> Map<K, V> immutableMapOf(K key1, V value1, K key2, V value2) {
    // we could use ImmutableMap from Guava for this - but need dependency
    LinkedHashMap<K, V> r = new LinkedHashMap<>();
    r.put(key1, value1);
    r.put(key2, value2);
    return Collections.unmodifiableMap(r);
  }
}
