package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.Nullable;

/**
 * <p>An interface that allows access to attribute values, loaded via the attribute system.</p>
 *
 * @param <K> type of the key - either {@code Long} for row-based values or {@link ItemIdentity} for item-based values.
 */
@PublicApi
public interface LoadedValues<K> {
  /**
   * <p>Returns the value loaded for the given key and attribute.</p>
   *
   * <p>This method may return {@code null} if the attribute and the key were not actually requested, or if the
   * value hasn't been loaded yet.
   * Note that {@code null} is not actually an attribute value, it's an absence of a value. It's not the same
   * as {@link AttributeValue#undefined()} (although they would be probably displayed in the same way).</p>
   *
   * @param key either row ID or item ID
   * @param attribute attribute
   * @param <T> type of the value
   * @return loaded value, or {@code null} if it is not present
   */
  @Nullable
  <T> LoadedValue<T> getLoadedValue(@Nullable K key, @Nullable AttributeSpec<T> attribute);

  /**
   * Returns the {@link AttributeValue} loaded for the given key and attribute.
   *
   * @param key either row ID or item ID
   * @param attribute attribute
   * @param <T> type of the value
   * @return attribute value, or {@code null} if it is not present
   */
  @Nullable
  default <T> AttributeValue<T> getAttributeValue(@Nullable K key, @Nullable AttributeSpec<T> attribute) {
    LoadedValue<T> value = getLoadedValue(key, attribute);
    return value == null ? null : value.getValue();
  }

  /**
   * <p>Checks if the value for the given key and attribute is outdated (up for recalculation).</p>
   *
   * <p>If the value for the given attribute and key is not present, this method returns {@code true}.
   * The absence of the value is considered outdated - loading this value should produce something
   * else instead of the null, at least {@link AttributeValue#undefined()}.
   * </p>
   *
   * @param key either row ID or item ID
   * @param attribute attribute
   * @return true if the value is outdated
   */
  default boolean isOutdated(@Nullable K key, @Nullable AttributeSpec<?> attribute) {
    LoadedValue<?> value = getLoadedValue(key, attribute);
    return value == null || value.isOutdated();
  }

  /**
   * Returns a value for the given row and attribute.
   *
   * @param key either row ID or item ID
   * @param attribute attribute
   * @param <T> type of the value
   * @return the value, or {@code null} if it is not present or undefined
   */
  @Nullable
  default <T> T get(@Nullable K key, @Nullable AttributeSpec<T> attribute) {
    AttributeValue<T> value = getAttributeValue(key, attribute);
    return value == null ? null : value.getValue();
  }
}
