package com.almworks.jira.structure.api.effect;

import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.util.I18nText;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;

/**
 * The result of {@link EffectProvider#resolve(StoredEffect) resolving}
 * an {@link StoredEffect effect description}. This could be a piece of
 * {@link Effect runnable code}, an error message, or an "empty" response,
 * indicating that no change is needed.
 */
@PublicApi
public class EffectResponse {
  @NotNull
  private final I18nText myDescription;

  @NotNull
  private final List<ItemIdentity> myAffectedItems;

  private EffectResponse(@NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
    myDescription = description;
    myAffectedItems = affectedItems;
  }

  /**
   * Returns the actual effect, or {@code null} if error or empty.
   */
  @Nullable
  public Effect getEffect() {
    return null;
  }

  /**
   * Returns the list of {@link ItemIdentity item IDs} affected by the change.
   * Could be empty.
   */
  @NotNull
  public List<ItemIdentity> getAffectedItems() {
    return myAffectedItems;
  }

  /**
   * <p>Returns a human-readable description of the change, phrased as a
   * possibility, e.g. "Assign FOO-123 to John Doe".
   *
   * <p>This description cannot be {@code null} even for error and empty
   * responses. The effect provider is required to produce the best possible
   * description, even for error responses (see {@link #error(I18nText, I18nText, List)}).
   */
  @NotNull
  public I18nText getDescription() {
    return myDescription;
  }

  /**
   * <p>Returns a human-readable description of the change, as if it already
   * happened, e.g. "Assigned FOO-123 to John Doe."
   *
   * <p>This description is {@code null} for error and empty responses.
   */
  @Nullable
  public I18nText getSuccessMessage() {
    return null;
  }

  /**
   * Describes why this effect cannot be applied, e.g. "Issue does not exist
   * or you have no permission to see it." Returns {@code null} for valid
   * and empty responses.
   */
  @Nullable
  public I18nText getError() {
    return null;
  }

  /**
   * Returns whether this response is an error, which means this change cannot
   * be applied.
   *
   * @see #getError()
   */
  public boolean isError() {
    return false;
  }

  /**
   * Returns whether this response is empty, which means that there are no
   * errors, but no change is needed, for example, because the item is already
   * in the required state.
   */
  public boolean isEmpty() {
    return false;
  }

  /**
   * Constructs a valid response, which means that the requested change
   * is both needed and possible.
   *
   * @param effect The actual runnable code that effects the change.
   * @param successMessage A human-readable description of the change as if
   *        it already happened, see {@link #getSuccessMessage()}.
   * @param description A human-readable description of the change as
   *        a possibility, see {@link #getDescription()}.
   * @param affectedItems The list of affected item IDs,
   *        see {@link #getAffectedItems()}.
   * @return a valid response
   */
  @NotNull
  public static EffectResponse valid(@NotNull Effect effect, @NotNull I18nText successMessage, @NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
    return new Valid(effect, successMessage, description, affectedItems);
  }

  /**
   * Constructs an error response, which means that the requested change
   * is impossible.
   *
   * @param error The error message describing why this change cannot
   *        be applied, see {@link #getError()}.
   * @param description A human-readable description of the change as
   *        a possibility, see {@link #getDescription()}. Note that this
   *        description cannot be {@code null} even for error responses.
   *        If an item mentioned in the effect description does not exist,
   *        the provider is supposed to use its internal identifier
   *        (e.g. issue ID) instead of the user-friendly identifier (issue key),
   *        but it still must produce a description.
   * @param affectedItems The list of affected item IDs,
   *        see {@link #getAffectedItems()}.
   * @return an error response
   */
  @NotNull
  public static EffectResponse error(@NotNull I18nText error, @NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
    return new Error(error, description, affectedItems);
  }

  /**
   * Constructs an empty response, which means that the requested change
   * is possible, but not required.
   *
   * @param description A human-readable description of the change as
   *        a possibility, see {@link #getDescription()}.
   * @param affectedItems The list of affected item IDs,
   *        see {@link #getAffectedItems()}.
   * @return an empty response.
   */
  @NotNull
  public static EffectResponse empty(@NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
    return new Empty(description, affectedItems);
  }


  /**
   * Represents a valid response, which means that the requested change
   * is both needed and possible.
   *
   * @see #valid(Effect, I18nText, I18nText, List)
   */
  public static final class Valid extends EffectResponse {
    @NotNull
    private final Effect myEffect;

    @NotNull
    private final I18nText mySuccessMessage;

    private Valid(@NotNull Effect effect, @NotNull I18nText successMessage, @NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
      super(description, affectedItems);
      myEffect = effect;
      mySuccessMessage = successMessage;
    }

    @NotNull
    @Override
    public Effect getEffect() {
      return myEffect;
    }

    @NotNull
    @Override
    public I18nText getSuccessMessage() {
      return mySuccessMessage;
    }
  }


  /**
   * Represents an error response, which means that the requested change
   * is impossible.
   *
   * @see #error(I18nText, I18nText, List)
   */
  public static final class Error extends EffectResponse {
    @NotNull
    private final I18nText myError;

    public Error(@NotNull I18nText error, @NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
      super(description, affectedItems);
      myError = error;
    }

    @NotNull
    @Override
    public I18nText getError() {
      return myError;
    }

    @Override
    public boolean isError() {
      return true;
    }
  }


  /**
   * Represents an empty response, which means that the requested change
   * is possible, but not required.
   *
   * @see #empty(I18nText, List)
   * @see #isEmpty()
   */
  public static final class Empty extends EffectResponse {
    public Empty(@NotNull I18nText description, @NotNull List<ItemIdentity> affectedItems) {
      super(description, affectedItems);
    }

    @Override
    public boolean isEmpty() {
      return true;
    }
  }
}
