package com.almworks.jira.structure.api.attribute;

import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

@Internal
public class AttributeSpecNormalization {
  /**
   * Does normalization based on the rules at https://dev.almworks.com/browse/STR-1464
   * See also AttributeSpecNormalization.js
   *
   * @param specParams
   * @return normalized specParams
   */
  @NotNull
  public static Map<String, Object> normalizeParams(@NotNull Map<String, Object> specParams) {
    if (!isNormalizationNeeded(specParams.values(), true)) return specParams;
    return normalizeMap(specParams);
  }

  @NotNull
  private static Map<String, Object> normalizeMap(@NotNull Map<String, Object> specParams) {
    if (specParams.isEmpty()) {
      return specParams;
    }
    LinkedHashMap<String, Object> r = new LinkedHashMap<>(specParams.size());
    for (Map.Entry<String, Object> e : specParams.entrySet()) {
      Object value = normalizeValue(e.getValue());
      if (!isValueRemoved(value)) {
        r.put(e.getKey(), value);
      }
    }
    return r;
  }

  private static List<Object> normalizeList(@NotNull List<Object> list) {
    if (list.isEmpty()) {
      return list;
    }
    ArrayList<Object> r = new ArrayList<>(list.size());
    for (Object value : list) {
      r.add(normalizeValue(value));
    }
    return r;
  }

  @Nullable
  @SuppressWarnings("unchecked")
  private static Object normalizeValue(Object value) {
    if (value instanceof Map) {
      return normalizeMap((Map<String, Object>) value);
    } else if (value instanceof List) {
      return normalizeList((List<Object>) value);
    }
    return value;
  }

  private static boolean isNormalizationNeeded(@NotNull Collection<Object> values, boolean canRemoveValue) {
    for (Object value : values) {
      if (isValueNormalizationNeeded(value, canRemoveValue)) return true;
    }
    return false;
  }

  @SuppressWarnings("unchecked")
  private static boolean isValueNormalizationNeeded(Object value, boolean canRemoveValue) {
    if (canRemoveValue && isValueRemoved(value)) return true;
    if (value instanceof List && isNormalizationNeeded((List<Object>) value, false)) return true;
    if (value instanceof Map && isNormalizationNeeded(((Map<String, Object>) value).values(), true)) return true;
    return false;
  }

  private static boolean isValueRemoved(Object value) {
    if (value == null) return true;
    if (value instanceof Boolean && !(boolean) value) return true;
    if (value instanceof Integer && (int)value == 0) return true;
    if (value instanceof Long && (long)value == 0L) return true;
    if (value instanceof List && ((List) value).isEmpty()) return true;
    if (value instanceof Map && ((Map) value).isEmpty()) return true;
    return false;
  }
}
