package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.attribute.loader.AttributeLoaderProvider;
import com.almworks.jira.structure.api.attribute.loader.ForestDependencyType;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collections;
import java.util.Set;

/**
 * <p>{@code AttributeTrail} carries information about what dependencies were used when calculating values for an
 * attribute. It can be used to correctly implement caching, which needs to invalidate a value if its recalculation
 * may provide a different result.</p>
 *
 * <p>Note that attribute trail may be different each time. Item dependencies may change based on how specific
 * value was calculated. And forest dependency type may change if a plugin with {@link AttributeLoaderProvider} is
 * enabled or disabled.
 *
 * @see StructureAttributeService
 * @see VersionedRowValues
 */
@PublicApi
public interface AttributeTrail {
  /**
   * Empty trail
   */
  AttributeTrail EMPTY = new EmptyTrail();

  /**
   * <p>Returns all types of in-forest dependencies that were used to calculate this value.</p>
   *
   * <p>An attribute value may be calculated using only the row requested ({@link ForestDependencyType#INDEPENDENT}),
   * or all sub-rows in the forest ({@link ForestDependencyType#AGGREGATE}), or in "top-down" manner
   * ({@link ForestDependencyType#PROPAGATE}), or a combination of these calculations.
   * Depending on the type, forest change and other items need to be
   * tracked for correct invalidation.</p>
   */
  @NotNull
  Set<ForestDependencyType> getTrailMode();

  /**
   * Returns a set of additional items that need to be tracked. If an item in the set is changed, then the value
   * for given row needs recalculation.
   *
   * @param row row ID for which the trail is retrieved
   * @return an item set which should be tracked for correct invalidation of the value for the given row, or
   * null if there's no value
   */
  @Nullable
  TrailItemSet getAdditionalItemDependencies(Long row);

  /**
   * Returns true if the value for the row can be reliably cached and invalidated, by observing forest changes
   * (according to trail mode), and item changes (according to row's item and additional dependencies).
   * {@code false} is returned only when the row
   * has been mapped to another row, and so correct invalidation according to forest updates is impossible
   */
  boolean isCacheable(Long row);


  /**
   * Empty trail implementation
   */
  class EmptyTrail implements AttributeTrail {
    @NotNull
    @Override
    public Set<ForestDependencyType> getTrailMode() {
      return Collections.emptySet();
    }

    @Nullable
    @Override
    public TrailItemSet getAdditionalItemDependencies(Long row) {
      return null;
    }

    @Override
    public boolean isCacheable(Long row) {
      return false;
    }
  }
}
