package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.attribute.loader.AttributeValue;
import com.almworks.jira.structure.api.darkfeature.DarkFeatures;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import com.google.common.collect.Sets;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * <p>Abstract class for defining a set of items (by their {@link ItemIdentity}).
 * Specific subclasses provide the different ways to define a set.</p>
 *
 * <p>{@code TrailItemSet} is used in {@link AttributeTrail} to provide a filter for item changes that should
 * cause invalidation of a calculated value.</p>
 *
 * <p>The implementations of this class are immutable and thread-safe.</p>
 *
 * <p>Implementation note: when adding another specific sub-class, make sure it is serializable and can be sent
 * over to the client code.</p>
 *
 * @see TrailItemSet.SpecificItems
 * @see OneType
 */
@PublicApi
public abstract class TrailItemSet {
  // todo unit tests

  // todo temporary setting - change to 10 when types are supported throughout, so far we need to stay with OneItem or SpecificItems
  private static final int SPECIFIC_ITEMS_LIMIT = DarkFeatures.getInteger("structure.trail.itemLimit", Integer.MAX_VALUE - 1);

  private static final int SPECIFIC_TYPES_LIMIT = DarkFeatures.getInteger("structure.trail.typeLimit", 10);

  private int myHashCode;

  /**
   * All specific sub-classes must be declared immediately in this file.
   */
  private TrailItemSet() {}

  /**
   * Checks if the set contains the given item.
   *
   * @param id item ID
   * @return true if the item is a part of this set
   */
  public abstract boolean contains(@Nullable ItemIdentity id);

  /**
   * <p>Expands the set to include the given item. The result of this operation is a new set, which a) includes
   * everything this set includes, b) includes given item.</p>
   *
   * <p>Note that the resulting set may contain <strong>more</strong> items, due to escalation to a more wide
   * set class. If you expand a set by a sufficient number of items, it will switch to be type-based set, which will
   * contain all items of the given types.</p>
   *
   * @param trailItem item to add to the set
   * @return a new set with all items from this set and with {@code trailItem}
   */
  @NotNull
  public abstract TrailItemSet expand(@Nullable ItemIdentity trailItem);

  /**
   * <p>Expands the set to include the given items. The result of this operation is a set, which includes
   * everything both this set and parameter set.</p>
   *
   * <p>Note that the resulting set may contain <strong>more</strong> items, due to escalation to a more wide
   * set class. If you expand a set by a sufficient number of items, it will switch to be type-based set, which will
   * contain all items of the given types.</p>
   *
   * @param trailItemSet item set to add to this set
   * @return a set with all items from this set and with {@code trailItem}
   */
  @NotNull
  public abstract TrailItemSet expand(@Nullable TrailItemSet trailItemSet);

  /**
   * Allows the caller to perform per-subclass actions.
   */
  public abstract void accept(@NotNull Visitor visitor);

  /**
   * Returns true if this set will not match any item.
   */
  public final boolean isEmpty() {
    return cardinality() == 0;
  }

  /**
   * <p>Internal method used for implementing equality between different sets.</p>
   *
   * <p>Returns the following number:</p>
   * <ul>
   * <li>0 means empty set.</li>
   * <li>Positive number (but not {@code Integer.MAX_VALUE}) means the number of specific items in the set.</li>
   * <li>Negative number means the set is type-based, and the absolute value means the number of types in the set.</li>
   * <li>{@code Integer.MAX_VALUE} means "all items" set.</li>
   * </ul>
   */
  abstract int cardinality();

  @Override
  public final boolean equals(Object obj) {
    if (!(obj instanceof TrailItemSet)) return false;
    TrailItemSet that = (TrailItemSet) obj;
    if (this.cardinality() != that.cardinality()) return false;
    return Collector.collect(this).equals(Collector.collect(that));
  }

  @Override
  public final int hashCode() {
    int hashCode = myHashCode;
    if (hashCode == 0) {
      hashCode = Collector.collect(this).hashCode();
      if (hashCode == 0) {
        hashCode = -1;
      }
      myHashCode = hashCode;
    }
    return hashCode;
  }

  @Override
  public final String toString() {
    return Collector.collect(this).toString();
  }

  /**
   * Constructs a set for specific item IDs.
   *
   * @param ids item identities - each element in the array must not be null!
   * @return a set that includes at least all the passed items
   */
  public static TrailItemSet of(ItemIdentity ... ids) {
    if (ids == null || ids.length == 0) {
      return None.NONE;
    } else if (ids.length == 1) {
      return new OneItem(ids[0]);
    } else if (ids.length <= SPECIFIC_ITEMS_LIMIT) {
      return new SpecificItems(Arrays.asList(ids));
    } else {
      return ofTypesGivenItems(Arrays.asList(ids), null);
    }
  }

  /**
   * Constructs a set for specific item types.
   *
   * @param types item types - each element in the array must not be null
   * @return a set that includes all items of the given types
   */
  public static TrailItemSet ofTypes(Collection<String> types) {
    if (types == null || types.isEmpty()) {
      return None.NONE;
    } else if (types.size() == 1) {
      return new OneType(types.iterator().next());
    } else if (types.size() <= SPECIFIC_TYPES_LIMIT) {
      return new SpecificTypes(types);
    } else {
      return AllItems.ALL_ITEMS;
    }
  }

  /**
   * Extracts trail item sets from a collection of attribute values and merges them into one.
   *
   * @param values attribute values - any element in the collection may be null
   * @return a set that includes all trail item sets from the given attribute values or null if there's no value with non-empty trail item set
   */
  @Nullable
  public static TrailItemSet fromValues(Collection<? extends AttributeValue<?>> values) {
    return values.stream()
      .filter(Objects::nonNull)
      .map(AttributeValue::getAdditionalDataTrail)
      .filter(trail -> !trail.isEmpty())
      .reduce(TrailItemSet::expand)
      .orElse(null);
  }

  private static TrailItemSet ofTypesGivenItems(Collection<ItemIdentity> items, @Nullable ItemIdentity oneMoreItem) {
    Stream<ItemIdentity> stream = items.stream();
    if (oneMoreItem != null) {
      stream = Stream.concat(stream, Stream.of(oneMoreItem));
    }
    Set<String> types = stream
      .map(id -> id == null ? null : id.getItemType())
      .filter(Objects::nonNull)
      .collect(Collectors.toSet());
    return ofTypes(types);
  }

  private static TrailItemSet mergeItemsToTypes(TrailItemSet typesSet, TrailItemSet itemsSet) {
    if (itemsSet instanceof OneItem) {
      OneItem oneItem = (OneItem) itemsSet;
      return typesSet.expand(oneItem.myItem);
    } else if (itemsSet instanceof SpecificItems) {
      SpecificItems items2 = (SpecificItems) itemsSet;
      return typesSet.expand(TrailItemSet.ofTypesGivenItems(items2.myItems, null));
    } else {
      assert itemsSet instanceof AllItems;
      return AllItems.ALL_ITEMS;
    }
  }

  /**
   * Represents an empty set.
   */
  public static final class None extends TrailItemSet {
    /**
     * An empty set
     */
    public static final TrailItemSet NONE = new None();

    @Override
    public boolean contains(ItemIdentity id) {
      return false;
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null) return this;
      return new OneItem(trailItem);
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      if (trailItemSet == null) return this;
      return trailItemSet;
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visitNone();
    }

    @Override
    int cardinality() {
      return 0;
    }
  }


  /**
   * Represents a set with just one item.
   */
  public static final class OneItem extends TrailItemSet {
    @NotNull
    private final ItemIdentity myItem;

    public OneItem(@NotNull ItemIdentity item) {
      if (item == null) throw new IllegalArgumentException("item cannot be null");
      myItem = item;
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && id.equals(myItem);
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || trailItem.equals(myItem)) return this;
      return new SpecificItems(Arrays.asList(myItem, trailItem));
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      if (trailItemSet == null || trailItemSet instanceof None) return this;
      return trailItemSet.expand(myItem);
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return 1;
    }

    @NotNull
    public ItemIdentity getItem() {
      return myItem;
    }
  }


  /**
   * Represents a set of several sepecific items.
   */
  public static final class SpecificItems extends TrailItemSet {
    @NotNull
    private final Set<ItemIdentity> myItems;

    public SpecificItems(Collection<ItemIdentity> items) {
      this(items.stream());
    }

    public SpecificItems(Stream<ItemIdentity> stream) {
      myItems = Collections.synchronizedSet(Collections.unmodifiableSet(stream.collect(Collectors.toSet())));
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return myItems.contains(id);
    }

    @NotNull
    public Set<ItemIdentity> getItems() {
      return myItems;
    }

    @Override
    @NotNull
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || myItems.contains(trailItem)) return this;
      if (myItems.size() >= SPECIFIC_ITEMS_LIMIT) {
        return ofTypesGivenItems(myItems, trailItem);
      }
      return new SpecificItems(Stream.concat(myItems.stream(), Stream.of(trailItem)));
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      if (trailItemSet == null || trailItemSet instanceof None) return this;
      if (trailItemSet instanceof OneItem) {
        OneItem oneItem = (OneItem) trailItemSet;
        return expand(Collections.singleton(oneItem.myItem));
      } else if (trailItemSet instanceof SpecificItems) {
        SpecificItems items = (SpecificItems) trailItemSet;
        return expand(items.myItems);
      } else if (trailItemSet instanceof OneType || trailItemSet instanceof SpecificTypes) {
        return TrailItemSet.ofTypesGivenItems(myItems, null).expand(trailItemSet);
      } else {
        return AllItems.ALL_ITEMS;
      }
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return myItems.size();
    }

    private TrailItemSet expand(Set<ItemIdentity> newItems) {
      Set<ItemIdentity> items = new HashSet<>(myItems);
      items.addAll(newItems);
      return items.size() > SPECIFIC_ITEMS_LIMIT ? ofTypesGivenItems(items, null) : new SpecificItems(items);
    }
  }


  /**
   * Represents a set of all items of one specific type.
   */
  public static final class OneType extends TrailItemSet {
    @NotNull
    private final String myItemType;

    public OneType(@NotNull String itemType) {
      if (itemType == null) throw new IllegalArgumentException("null itemType");
      myItemType = itemType;
    }

    @NotNull
    public String getItemType() {
      return myItemType;
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && id.getItemType().equals(myItemType);
    }

    @Override
    @NotNull
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null) return this;
      String itemType = trailItem.getItemType();
      if (itemType.equals(myItemType)) return this;
      return new SpecificTypes(Arrays.asList(myItemType, itemType));
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      if (trailItemSet == null || trailItemSet instanceof None) return this;
      if (trailItemSet instanceof OneType) {
        OneType oneType = (OneType) trailItemSet;
        if (oneType.myItemType.equals(myItemType)) return this;
        return ofTypes(Sets.newHashSet(myItemType, oneType.myItemType));
      } else if (trailItemSet instanceof SpecificTypes) {
        SpecificTypes types = (SpecificTypes) trailItemSet;
        return types.expand(Collections.singleton(myItemType));
      } else {
        return TrailItemSet.mergeItemsToTypes(this, trailItemSet);
      }
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return -1;
    }
  }


  /**
   * Represents a set of all items of several specific types.
   */
  public static final class SpecificTypes extends TrailItemSet {
    @NotNull
    private final Set<String> myTypes;

    public SpecificTypes(Collection<String> types) {
      this(types.stream());
    }

    public SpecificTypes(Stream<String> stream) {
      myTypes = Collections.synchronizedSet(Collections.unmodifiableSet(stream.collect(Collectors.toSet())));
    }

    @Override
    public boolean contains(ItemIdentity id) {
      return id != null && myTypes.contains(id.getItemType());
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable ItemIdentity trailItem) {
      if (trailItem == null || myTypes.contains(trailItem.getItemType())) return this;
      if (myTypes.size() >= SPECIFIC_TYPES_LIMIT) {
        return AllItems.ALL_ITEMS;
      }
      return new SpecificTypes(Stream.concat(myTypes.stream(), Stream.of(trailItem.getItemType())));
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      if (trailItemSet == null || trailItemSet instanceof None) return this;
      if (trailItemSet instanceof OneType) {
        OneType oneType = (OneType) trailItemSet;
        if (myTypes.contains(oneType.myItemType)) return this;
        return expand(Collections.singleton(oneType.myItemType));
      } else if (trailItemSet instanceof SpecificTypes) {
        SpecificTypes plusTypes = (SpecificTypes) trailItemSet;
        if (myTypes.containsAll(plusTypes.myTypes)) return this;
        return expand(plusTypes.myTypes);
      } else {
        return TrailItemSet.mergeItemsToTypes(this, trailItemSet);
      }
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visit(this);
    }

    @Override
    int cardinality() {
      return -myTypes.size();
    }

    @NotNull
    public Set<String> getTypes() {
      return myTypes;
    }

    private TrailItemSet expand(Set<String> newTypes) {
      Set<String> types = new HashSet<>(myTypes);
      types.addAll(newTypes);
      return types.size() > SPECIFIC_TYPES_LIMIT ? AllItems.ALL_ITEMS : new SpecificTypes(types);
    }
  }


  /**
   * Represents a set of all items.
   */
  public static final class AllItems extends TrailItemSet {
    public static final TrailItemSet ALL_ITEMS = new AllItems();

    @Override
    public boolean contains(ItemIdentity id) {
      return true;
    }

    @NotNull
    @Override
    public TrailItemSet expand(ItemIdentity trailItem) {
      return this;
    }

    @NotNull
    @Override
    public TrailItemSet expand(@Nullable TrailItemSet trailItemSet) {
      return this;
    }

    @Override
    public void accept(@NotNull Visitor visitor) {
      visitor.visitAll();
    }

    @Override
    int cardinality() {
      return Integer.MAX_VALUE;
    }
  }


  /**
   * Visitor interface for analyzing the set.
   */
  public interface Visitor {
    void visitNone();

    void visit(OneItem oneItem);
    void visit(SpecificItems set);
    void visit(OneType set);
    void visit(SpecificTypes specificTypes);

    void visitAll();
  }


  /**
   * Alternate visitor interface for reading out the specific items and types.
   */
  public interface ReadVisitor extends Visitor {
    void visitItem(ItemIdentity item);
    void visitType(String type);

    @Override
    default void visitNone() {
    }

    @Override
    default void visit(OneItem set) {
      visitItem(set.getItem());
    }

    @Override
    default void visit(SpecificItems set) {
      for (ItemIdentity item : set.getItems()) {
        visitItem(item);
      }
    }

    @Override
    default void visit(OneType set) {
      visitType(set.getItemType());
    }

    @Override
    default void visit(SpecificTypes set) {
      for (String type : set.getTypes()) {
        visitType(type);
      }
    }
  }


  /**
   * Used to collect specific types and items stored in the TrailItemSet. Not thread-safe.
   */
  public static class Collector implements ReadVisitor {
    private Set<String> myTypes;
    private Set<ItemIdentity> myItems;
    private boolean myAll;

    private Collector(int cardinality) {
      if (cardinality > 0 && cardinality < Integer.MAX_VALUE) {
        myItems = new HashSet<>(cardinality);
      } else if (cardinality < 0) {
        myTypes = new HashSet<>(-cardinality);
      }
    }

    public static Collector collect(TrailItemSet set) {
      Collector collector = new Collector(set.cardinality());
      set.accept(collector);
      return collector;
    }

    public Set<String> getTypes() {
      return myTypes;
    }

    public Set<ItemIdentity> getItems() {
      return myItems;
    }

    public boolean isAll() {
      return myAll;
    }

    @Override
    public void visitItem(ItemIdentity item) {
      if (myItems == null) {
        assert false : "set reported bad cardinality [" + item + "]";
        myItems = new HashSet<>();
      }
      myItems.add(item);
    }

    @Override
    public void visitType(String type) {
      if (myTypes == null) {
        assert false : "set reported bad cardinality [" + type + "]";
        myTypes = new HashSet<>();
      }
      myTypes.add(type);
    }

    @Override
    public void visitAll() {
      myAll = true;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Collector collector = (Collector) o;
      return myAll == collector.myAll &&
        Objects.equals(myTypes, collector.myTypes) &&
        Objects.equals(myItems, collector.myItems);
    }

    @Override
    public int hashCode() {
      return Objects.hash(myTypes, myItems, myAll);
    }

    @Override
    public String toString() {
      if (myAll) return "{ALL}";
      String types = myTypes == null ? "" : StringUtils.join(myTypes, ',');
      String items = myItems == null ? "" : StringUtils.join(myItems, ',');
      String comma = types != null && items != null ? "," : "";
      return "{" + types + comma + items + "}";
    }
  }
}
