package com.almworks.jira.structure.api.attribute;

import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.item.ItemTracker;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

/**
 * <p>This interface extends the basic {@code RowValues} interface, which allows retrieving values calculated by
 * {@link StructureAttributeService}. In addition to the basic {@link RowValues#get(Long, AttributeSpec)} method,
 * this interface provides information about the version of the forest and version of the items that were current
 * when the values were calculated.</p>
 *
 * <p>Also, the interface provides a way to retrieve {@link AttributeTrail} for each requested attribute, which
 * is needed to correctly implement value caching.</p>
 *
 * @see RowValues
 * @see StructureAttributeService
 */
@PublicApi
public interface VersionedRowValues extends RowValues {
  /**
   * Constant empty result
   */
  VersionedRowValues EMPTY = new EmptyVersionedRowValues();

  /**
   * <p>Returns the version of the forest (defined by {@link ForestSpec}), which was used to calculate the values.
   * Forest version is important for "aggregate" and "propagate" values.</p>
   *
   * <p>When values are requested for arbitrary forest (not defined by a {@code ForestSpec}), this method
   * always returns {@link DataVersion#ZERO}.</p>
   */
  @NotNull
  DataVersion getForestVersion();

  /**
   * Returns the global version of items, which was current when the values were calculated. When items change,
   * the global items version increases, so it can be used to implement caching with invalidation.
   *
   * @see ItemTracker
   */
  @NotNull
  DataVersion getItemsVersion();

  /**
   * Empty result implementation
   */
  final class EmptyVersionedRowValues implements VersionedRowValues {
    @NotNull
    public DataVersion getForestVersion() {
      return DataVersion.ZERO;
    }

    @NotNull
    public DataVersion getItemsVersion() {
      return DataVersion.ZERO;
    }

    @NotNull
    public AttributeTrail getTrail(AttributeSpec<?> attribute) {
      return AttributeTrail.EMPTY;
    }

    public <T> T get(Long row, AttributeSpec<T> attribute) {
      return null;
    }
  }
}
