package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.export.*;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;

/**
 * <p>An {@code AttributeContext} is associated with every attribute request and Structure export request. The context
 * contains some basic information, like the current user and JIRA base URL, and it can also contain other data
 * associated with the current request by Structure or other plug-ins. The current context is passed into every call
 * to {@link AttributeLoaderProvider AttributeLoaderProvider}, {@link AttributeLoader AttributeLoader},
 * {@link ExportRendererProvider ExportRendererProvider}, and {@link ExportRenderer ExportRenderer}.</p>
 *
 * <p>Please note that the context key space is shared by all plug-ins, so we advise you to use plugin-private keys
 * (e.g. singleton objects or custom enums) to avoid conflicts.</p>
 *
 * @see ExportContextKeys
 * @see ExportRequestContext
 * @see ExportRenderContext
 */
@PublicApi
public interface AttributeContext {
  /**
   * @return The logged-in user, or {@code null} if anonymous.
   */
  @Nullable
  ApplicationUser getUser();

  /**
   * @return The {@link I18nHelper} instance for the current user.
   */
  @NotNull
  I18nHelper getI18nHelper();

  /**
   * @return The current user's {@link Locale}.
   */
  @NotNull
  Locale getLocale();

  /**
   * @return The base URL of the JIRA instance.
   */
  @NotNull
  String getBaseUrl();

  /**
   * <p>Associate an arbitrary {@code value} with the given request. The value can be retrieved by passing the same
   * {@code key} to {@link #getObject(Object)}.</p>
   *
   * <p>Please note that {@code AttributeContext} instances are shared by all
   * {@link AttributeLoaderProvider attribute loader providers} or
   * {@link ExportRendererProvider export renderer providers} from
   * all plug-ins. Therefore we recommend you to use plugin-private keys (e.g. singleton objects or custom enums) to
   * avoid conflicts.</p>
   *
   * @param key The key.
   * @param value The value. Pass {@code null} to clear the object associated with the given {@code key}.
   * @throws IllegalArgumentException if {@code key} is read-only.
   */
  void putObject(@NotNull Object key, @Nullable Object value);

  /**
   * Retrieve the value associated with a key.
   * @param key The key.
   * @param <T> The expected type of the value.
   * @return The object associated with the {@code key}, or {@code null} if none.
   * @throws ClassCastException if the value is not an instance of type {@code <T>}.
   */
  @Nullable
  <T> T getObject(@NotNull Object key);

  /**
   * Returns the base forest spec, for which the attribute is being calculated. Returns null when
   * the attribute is calculated for a forest without forest spec.
   */
  @Nullable
  ForestSpec getBaseForestSpec();

  default long getBaseStructureId() {
    ForestSpec forestSpec = getBaseForestSpec();
    return forestSpec == null ? 0 : StructureUtil.nnl(forestSpec.getStructureId());
  }
}
