package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.row.StructureRow;
import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.List;
import java.util.Set;

@PublicSpi
public interface AttributeLoader<T> {
  AttributeSpec<T> getAttributeSpec();

  boolean isEveryItemTypeSupported();

  boolean isItemTypeSupported(String itemType);

  // defines which other attributes must be calculated for row X, before calling getFunction().loadValue()
  @NotNull
  Set<? extends AttributeSpec<?>> getAttributeDependencies();

  AttributeCachingStrategy getCachingStrategy();

  @PublicSpi
  interface ForestIndependent<T> extends AttributeLoader<T> {
    AttributeValue<T> loadValue(StructureRow row, Context context);
  }

  @PublicSpi
  interface Aggregate<T> extends AttributeLoader<T> {
    // childrenValues does not contain nulls
    // todo the above is not provable, make contract more lenient
    AttributeValue<T> loadValue(List<AttributeValue<T>> childrenValues, AggregateContext<T> context);
  }

  @PublicSpi
  interface Propagate<T> extends AttributeLoader<T> {
    // Returns a list that is parallel to the children list
    // `null` is treated as "list of undefined"
    @Nullable
    List<AttributeValue<T>> loadChildrenValues(@Nullable AttributeValue<T> rowValue, List<StructureRow> children,
      PropagateContext<T> context);
  }

  @PublicApi
  interface Context extends AttributeContext {
    // returns the current row
    @NotNull
    StructureRow getRow();
  
    // get cached value of attribute dependency for the current row
    @Nullable
    <V> V getValue(AttributeSpec<V> dependency);

    @Nullable
    <V> AttributeValue<V> getAttributeValue(AttributeSpec<V> dependency);
  }

  @PublicApi
  interface AggregateContext<T> extends Context {
    List<StructureRow> getChildren();

    T getChildValue(StructureRow child);

    AttributeValue<T> getChildAttributeValue(StructureRow child);
  }

  @PublicApi
  interface PropagateContext<T> extends Context {
    // returns cached value for the current row
    T getValue();

    AttributeValue<T> getAttributeValue();

    List<StructureRow> getChildren();

    // returns cached value of attribute dependency for the specified child row 
    <V> V getChildValue(StructureRow child, AttributeSpec<V> dependency);

    <V> AttributeValue<V> getChildAttributeValue(StructureRow child, AttributeSpec<V> dependency);
  }
}
