package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.error.StructureProviderException;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

// todo problem (STR-504): suppose we register loaders for foo:B, foo:C and we're asked for foo:A
// todo   where C can be converted to A, but B can't
// todo   we'll return loader foo:B, but should have returned foo:C
@Internal
public abstract class SimpleAttributeProvider implements AttributeLoaderProvider {
  // attributeId => loaders
  private final Map<String, List<AttributeLoader<?>>> myLoaders = new HashMap<>();

  // Call from constructor
  protected final void registerLoader(AttributeLoader<?> loader) {
    AttributeSpec<?> spec = loader.getAttributeSpec();
    myLoaders.computeIfAbsent(spec.getId(), id -> new ArrayList<>(2)).add(loader);
  }

  @SafeVarargs
  protected final <T> void registerCompositeLoader(AttributeSpec<T> spec, AttributeLoader<T>... loaders) {
    registerLoader(CompositeAttributeLoader.create(spec, Arrays.asList(loaders)));
  }

  @Nullable
  public synchronized AttributeLoader<?> createAttributeLoader(
    @NotNull AttributeSpec<?> spec, @NotNull AttributeContext context) throws StructureProviderException
  {
    List<AttributeLoader<?>> loaders = myLoaders.get(spec.getId());
    if (loaders == null) {
      return null;
    }
    return loaders.stream()
      .filter(l -> StructureUtil.isSubMap(spec.getParamsMap(), l.getAttributeSpec().getParamsMap()))
      .sorted(Comparator.comparingInt(
        (AttributeLoader<?> l) -> -l.getAttributeSpec().getParamsMap().size())
          .thenComparingInt(l -> l.getAttributeSpec().is(spec.getFormat()) ? 0 : 1))
      .findFirst().orElse(null);
  }
}
