package com.almworks.jira.structure.api.attribute.loader.basic;

import com.almworks.jira.structure.api.attribute.loader.AttributeValue;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.math.*;

public class NumberAccumulator {
  private static final MathContext MC = MathContext.DECIMAL64;
  
  private long myLongValue;
  private double myDoubleValue;
  private boolean myDefined;
  private boolean myFloating;

  // if not null, this is the holder of the value, ignore everything else
  private BigDecimal myBigDecimal;

  // Compensation by Kahan summation
  private double myDoubleSumCompensation = 0.0;
  private BigDecimal myBigDecimalCompensation = BigDecimal.ZERO;

  public void add(Number value) {
    addMultiply(value, 1);
  }

  public void remove(Number value) {
    addMultiply(value, -1);
  }

  public void removeMultiply(Number value, int count) {
    addMultiply(value, -count);
  }

  public void addMultiply(Number value, int count) {
    if (value == null) {
      return;
    }
    myDefined = true;

    BigDecimal bigValue = null;
    double doubleValue = 0.0;
    long longValue = 0;
    boolean integral = false;

    if (value instanceof BigDecimal) {
      bigValue = (BigDecimal) value;
    } else if (value instanceof BigInteger) {
      bigValue = new BigDecimal((BigInteger) value);
    } else {
      doubleValue = value.doubleValue();
      longValue = value.longValue();
      // arguable point about how to detect float from integral type - we could actually check for the specific class of the Number we have
      // however, there are many of them, and this seems faster; in the end the provided sum will be checked either for longValue or doubleValue
      integral = doubleValue == (double) longValue;
    }

    if (bigValue != null && myBigDecimal == null) {
      // switch to big decimal
      switchToBigDecimal();
    }

    if (myBigDecimal != null) {
      addBigDecimal(bigValue, doubleValue, longValue, integral, count);
    } else {
      addPrimitive(doubleValue, longValue, integral, count);
    }
  }

  private void addPrimitive(double doubleValue, long longValue, boolean integral, int count) {
    if (!integral && !myFloating) {
      myFloating = true;
      myLongValue = 0; // cleanup
    }

    /*
     * Double value is calculated always, long value is calculated only when we still have integer numbers.
     * The reason is that long to double conversion is not always loss-less.
     */

    doubleValue = doubleValue * count;
    double add = doubleValue - myDoubleSumCompensation;
    double sum = myDoubleValue + add;
    myDoubleSumCompensation = (sum - myDoubleValue) - add;
    myDoubleValue = sum;

    if (!myFloating) {
      // Accumulate long values unless we have encountered a floating-point value -- then longs are not needed anymore.
      myLongValue += longValue * count;
    }
  }

  private void addBigDecimal(BigDecimal bigValue, double doubleValue, long longValue, boolean integral, int count) {
    if (bigValue == null) {
      if (integral) {
        bigValue = new BigDecimal(longValue);
      } else {
        bigValue = new BigDecimal(doubleValue, MC);
      }
    }
    if (count != 1) {
      bigValue = bigValue.multiply(new BigDecimal(count), MC);
    }

    BigDecimal add = bigValue.subtract(myBigDecimalCompensation, MC);
    BigDecimal sum = myBigDecimal.add(add, MC);
    myBigDecimalCompensation = sum.subtract(myBigDecimal, MC).subtract(add, MC);
    myBigDecimal = sum;
  }

  private void switchToBigDecimal() {
    myBigDecimal = myFloating ? new BigDecimal(myDoubleValue, MC) : new BigDecimal(myLongValue);
    myLongValue = 0;
    myDoubleValue = 0.0;
    myFloating = false;
  }

  @NotNull
  public AttributeValue<Number> toValue() {
    Number number = toNumber();
    return number == null ? AttributeValue.undefined() : AttributeValue.of(number);
  }

  @Nullable
  public Number toNumber() {
    if (!myDefined) {
      return null;
    }
    if (myBigDecimal == null) {
      // Note: important not to use ?: here!
      if (myFloating) {
        return myDoubleValue;
      } else {
        return myLongValue;
      }
    } else {
      // Strip trailing zeros to achieve equality with the same-valued AttributeValues.
      return myBigDecimal.stripTrailingZeros();
    }
  }

  @Override
  public String toString() {
    String val;
    if (!myDefined) {
      val = "";
    } else if (myBigDecimal != null) {
      val = myBigDecimal.stripTrailingZeros().toPlainString();
    } else {
      val = myFloating ? myDoubleValue + "D" : myLongValue + "L";
    }
    return "NumberAccumulator[" + val + "]";
  }
}