package com.almworks.jira.structure.api.error;

import com.almworks.jira.structure.api.forest.ForestSource;
import com.atlassian.annotations.PublicApi;

import static com.almworks.jira.structure.api.error.StructureErrorCategory.*;

/**
 * <p>{@code StructureErrors} is a enumeration of all public Structure errors.</p>
 *
 * @see StructureError
 * @see StructureException
 */
@PublicApi
@StructureError.CodeRange(from = 0, to = 6999)
public enum StructureErrors implements StructureError {
  /*
   * Implementation note for ALM Works about picking a new error code.
   * Previous version of StructureError (in Structure 2) did not have categories, so the error code range gave
   * roughly the same information about the type of error. Now it is not relevant because we have category, so we
   * can pick any error code within reserved range.
   *
   * However, to maintain the same style, it is recommended to pick a value that is close to a similar error code.
   */

  /**
   * <p>Non-specific error. The meaning can be derived only from the error message and other parameters.</p>
   *
   * <p>Implementation note: try to avoid using it and use a specific one instead.</p>
   */
  GENERIC_ERROR(1000, GENERAL),

  /**
   * Problem with Structure license or extension license, which prevents operation from being executed.
   */
  LICENSE_PROBLEM(1003, GENERAL),

  /**
   * Internal problem with Structure locking mechanism.
   */
  LOCK_FAILED(1004, INTERNAL),

  /**
   * Invalid parameter used (either in operation or in settings of a synchronizer or generator or other entity)
   */
  INVALID_PARAMETER(2400, INVALID_INPUT),

  /**
   * Indicates that there are additional decisions that need to be made by the user to execute the operations.
   * Thrown from UpdatableForestSource.update()
   */
  DECISION_REQUIRED(2401, GENERAL),

  /**
   * Generic update error for actions applied to a forest source. Includes generic failures caused by
   * forest configuration and unexpected failures from trying to apply effects to items.
   */
  FOREST_SOURCE_ACTION_FAILED(2402, GENERAL),

  /**
   * Item update failed. The related item can be retrieved from {@link StructureException#getItem()}.
   */
  UPDATE_ITEM_FAILED(2403, GENERAL),

  /**
   * Structure attempted to create a new item but something failed in the corresponding module.
   */
  CREATE_ITEM_FAILED(2404, GENERAL),

  /**
   * When a structure has "requires editable parent" flag on, and the attempted operation is under an item that
   * the user does not have the edit access to.
   */
  FOREST_CHANGE_PROHIBITED_BY_PARENT_PERMISSIONS(4002, NO_PERMISSION),

  /**
   * Trying to do something with an issue from a project that is not enabled for Structure.
   */
  PROJECT_NOT_ENABLED_FOR_STRUCTURE(4003, NO_PERMISSION),

  /**
   * Trying to access a structure that does not exist, or the user has no read access to it.
   */
  STRUCTURE_NOT_EXISTS_OR_NOT_ACCESSIBLE(4005, NOT_FOUND),

  /**
   * The user does not have Create Structure permission.
   */
  NOT_ALLOWED_TO_CREATE_STRUCTURE(4006, NO_PERMISSION),

  /**
   * The user does not have Control access level to the structure.
   */
  STRUCTURE_ADMIN_DENIED(4007, NO_PERMISSION),

  /**
   * The user does not have Edit access level to the structure.
   */
  STRUCTURE_EDIT_DENIED(4008, NO_PERMISSION),

  /**
   * The user does not have JIRA Administrators permission.
   */
  JIRA_ADMIN_DENIED(4009, NO_PERMISSION),

  /**
   * The user does not have access to Structure plugin functionality.
   */
  STRUCTURE_PLUGIN_ACCESS_DENIED(4010, NO_PERMISSION),

  /**
   * Trying to edit archived structure.
   */
  ARCHIVED_STRUCTURE_EDIT_DENIED(4012, NO_PERMISSION),

  /**
   * Trying to change structure's owner while not being an admin.
   */
  STRUCTURE_OWNER_EDIT_DENIED(4013, NO_PERMISSION),

  /**
   * The user does not have Edit Generators access level to the structure.
   */
  STRUCTURE_GENERATORS_EDIT_DENIED(4014, NO_PERMISSION),

  /**
   * Inaccessible view.
   */
  VIEW_NOT_EXISTS_OR_NOT_ACCESSIBLE(4101, NOT_FOUND),

  /**
   * The user does not have Control permission on the view.
   */
  VIEW_ADMIN_DENIED(4102, NO_PERMISSION),

  /**
   * The user does not have Edit permission on the view.
   */
  VIEW_EDIT_DENIED(4103, NO_PERMISSION),

  /**
   * The user does not have permissions to create views.
   */
  VIEW_CREATION_DENIED(4104, NO_PERMISSION),

  /**
   * The user does not have permissions to create perspectives.
   */
  PERSPECTIVE_CREATION_DENIED(4201, NO_PERMISSION),

  /**
   * Attempting to edit an item without permission. When it comes to issues, the error may be reported not only
   * when the user does not have permissions, but when JIRA prohibits editing the issue based on workflow.
   */
  ITEM_EDIT_DENIED(4403, NO_PERMISSION),

  /**
   * The user does not have access to automation feature.
   */
  AUTOMATION_ACCESS_DENIED(4500, NO_PERMISSION),

  /**
   * The background job is not available.
   */
  JOB_NOT_EXISTS(4603, NOT_FOUND),

  /**
   * Cannot access an item because either it is missing or the user does not have access to it. Specific
   * item must be supplied via {@link StructureException#getItem()}.
   */
  ITEM_NOT_EXISTS_OR_NOT_ACCESSIBLE(4700, NOT_FOUND),

  /**
   * Invalid under/after/before coordinates used to change the forest, or other parameters used to make a forest operation.
   * The error can be fixed by changing the parameters of the forest operation.
   */
  INVALID_FOREST_OPERATION(5002, INVALID_INPUT),

  /**
   * Unsupported operation on {@link ForestSource} &mdash; for example, trying to make changes on a read-only
   * source. The error cannot be fixed on the same forest source.
   */
  UNSUPPORTED_FOREST_SOURCE_OPERATION(5003, INVALID_INPUT),

  /**
   * An extension module (item type, attribute loader, etc) is expected to exist in the system but was missing.
   */
  UNAVAILABLE_MODULE(5004, GENERAL),

  /**
   * An extension module is throwing an exception or fails to get linked to.
   */
  MODULE_ERROR(5005, GENERAL),

  /**
   * Signature has changed and the client needs to reload the initial data before applying updates.
   */
  SIGNATURE_MISMATCH(5006, GENERAL),

  /**
   * Automation feature failed.
   */
  AUTOMATION_FAILED(5007, GENERAL);

  private final int myCode;
  private final StructureErrorCategory myCategory;

  StructureErrors(int code, StructureErrorCategory category) {
    if (category == null) throw new IllegalArgumentException("category must not be null");
    myCode = code;
    myCategory = category;
  }

  @Override
  public int getCode() {
    return myCode;
  }

  @Override
  public StructureErrorCategory getCategory() {
    return myCategory;
  }

  public String toString() {
    return super.toString() + "[" + myCode + "]";
  }
}
