package com.almworks.jira.structure.api.forest.item;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.forest.raw.ArrayForest;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.item.ItemResolver;
import com.almworks.jira.structure.api.row.SimpleRow;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.util.JiraComponents;
import com.almworks.jira.structure.api.util.RowTree;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.NotThreadSafe;
import java.util.HashMap;
import java.util.Map;

/**
 * Fat builder - uses memory-hungry but fast RowTree.
 */
@PublicApi
@NotThreadSafe
public final class ItemForestBuilderImpl implements ItemForestBuilder<ItemForestBuilderImpl> {
  private final ItemResolver myItemResolver = JiraComponents.getOSGiComponentInstanceOfType(ItemResolver.class);

  private final RowTree myTree = new RowTree();
  private final Map<Long, StructureRow> myRows = new HashMap<>();

  private long myNextRowId = -1;
  private RowTree.Node myCurrentParent = myTree.getSuperRoot();
  private RowTree.Node myCurrentRow = null;

  @NotNull
  @Override
  public ItemForestBuilderImpl nextRow(@NotNull ItemIdentity itemId) {
    return nextRow0(itemId, 0);
  }

  @NotNull
  @Override
  public ItemForestBuilderImpl nextRow(@NotNull ItemIdentity itemId, long semantics) {
    return nextRow0(itemId, semantics);
  }

  @NotNull
  private ItemForestBuilderImpl nextRow0(@NotNull ItemIdentity itemId, long semantics) {
    long rowId = myNextRowId--;
    StructureRow row = new SimpleRow(rowId, itemId, semantics, myItemResolver);
    myRows.put(rowId, row);
    myCurrentRow = myTree.insertNode(rowId, 0, myCurrentParent, myCurrentRow);
    return this;
  }

  @NotNull
  @Override
  public ItemForestBuilderImpl nextLevel() {
    if (myCurrentRow == null) {
      throw new IllegalStateException();
    }
    myCurrentParent = myCurrentRow;
    myCurrentRow = null;
    return this;
  }

  @NotNull
  @Override
  public ItemForestBuilderImpl prevLevel() {
    if (myCurrentRow == null) {
      throw new IllegalStateException();
    }
    if (myCurrentParent == myTree.getSuperRoot()) {
      throw new IllegalStateException();
    }
    myCurrentRow = myCurrentRow.getParent();
    myCurrentParent = myCurrentRow.getParent();
    return this;
  }

  @NotNull
  @Override
  public ItemForestBuilderImpl nextFragment(@NotNull ItemForest fragment) {
    Forest forest = fragment.getForest();
    if (forest.isEmpty()) {
      return this;
    }
    LongArray rows = new LongArray();
    for (LongIterator it : forest.getRows()) {
      StructureRow row = fragment.getRow(it.value());
      long rowId = myNextRowId--;
      myRows.put(rowId, new SimpleRow(rowId, row.getItemId(), row.getSemantics(), myItemResolver));
      rows.add(rowId);
    }
    assert myCurrentRow == null || myCurrentRow.getLastChild() == null : "fixme - will set myCurrentRow incorrectly";
    ArrayForest copyForest = new ArrayForest(rows, new IntArray(forest.getDepths()), true);
    myTree.appendForest(copyForest, myCurrentParent, myCurrentRow);
    myCurrentRow = myCurrentParent.getLastChild();
    return this;
  }

  @NotNull
  public ItemForest build() {
    return ImmutableItemForest.of(myTree.toForest(), myRows);
  }
}
