package com.almworks.jira.structure.api.item;

import com.almworks.jira.structure.api.util.JiraFunc;
import com.atlassian.annotations.PublicApi;
import com.atlassian.crowd.embedded.api.Group;
import com.atlassian.jira.bc.project.component.ProjectComponent;
import com.atlassian.jira.issue.Issue;
import com.atlassian.jira.issue.customfields.option.Option;
import com.atlassian.jira.issue.issuetype.IssueType;
import com.atlassian.jira.issue.label.Label;
import com.atlassian.jira.issue.priority.Priority;
import com.atlassian.jira.issue.resolution.Resolution;
import com.atlassian.jira.issue.status.Status;
import com.atlassian.jira.project.Project;
import com.atlassian.jira.project.version.Version;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;

/**
 * <p>This class contains utility methods for checking and creating {@link ItemIdentity}.</p>
 *
 * <p>Typical use cases:</p>
 *
 * <pre>
 *   ItemIdentity itemId = CoreIdentities.issue(10000);
 *   ...
 *   if (CoreIdentities.isIssue(itemId)) {
 *     long issueId = itemId.getLongId();
 *   }
 * </pre>
 */
@PublicApi
public class CoreIdentities {
  public static final String ANONYMOUS_USER_ID = "-";

  public static final ItemIdentity NEW_ISSUE = issue(Long.MAX_VALUE);
  public static final ItemIdentity NEW_FOLDER = folder(Long.MAX_VALUE);
  public static final ItemIdentity NEW_GENERATOR = generator(Long.MAX_VALUE);

  public static final ItemIdentity MISSING_ITEM = ItemIdentity.longId(CoreItemTypes.MISSING, 0);

  private CoreIdentities() {
  }

  public static ItemIdentity user(@Nullable ApplicationUser user) {
    return user(user == null ? null : user.getKey());
  }

  public static ItemIdentity user(@Nullable String userKey) {
    return ItemIdentity.stringId(CoreItemTypes.USER, userKey == null ? ANONYMOUS_USER_ID : userKey);
  }

  public static ItemIdentity group(@NotNull String groupId) {
    return ItemIdentity.stringId(CoreItemTypes.GROUP, groupId);
  }

  public static ItemIdentity group(@NotNull Group group) {
    return group(group.getName());
  }

  public static ItemIdentity project(Project value) {
    return project(value.getId());
  }

  public static ItemIdentity project(long projectId) {
    return ItemIdentity.longId(CoreItemTypes.PROJECT, projectId);
  }

  public static ItemIdentity priority(Priority value) {
    return ItemIdentity.stringId(CoreItemTypes.PRIORITY, value.getId());
  }

  public static ItemIdentity status(Status value) {
    return ItemIdentity.stringId(CoreItemTypes.STATUS, value.getId());
  }

  public static ItemIdentity resolution(Resolution value) {
    return ItemIdentity.stringId(CoreItemTypes.RESOLUTION, value.getId());
  }

  public static ItemIdentity label(Label value) {
    return ItemIdentity.stringId(CoreItemTypes.LABEL, value.getLabel());
  }

  public static ItemIdentity issueType(IssueType value) {
    return ItemIdentity.stringId(CoreItemTypes.ISSUETYPE, value.getId());
  }

  public static ItemIdentity issue(long issueId) {
    return ItemIdentity.longId(CoreItemTypes.ISSUE, issueId);
  }

  public static ItemIdentity issue(Issue issue) {
    return issue(issue.getId());
  }

  public static ItemIdentity version(@NotNull Version version) {
    return ItemIdentity.longId(CoreItemTypes.VERSION, version.getId());
  }

  public static ItemIdentity version(long id) {
    return ItemIdentity.longId(CoreItemTypes.VERSION, id);
  }

  public static ItemIdentity versionName(String name) {
    return ItemIdentity.stringId(CoreItemTypes.VERSION_NAME, canonicalVersionName(name));
  }

  public static ItemIdentity versionName(@NotNull Version version) {
    return ItemIdentity.stringId(CoreItemTypes.VERSION_NAME, JiraFunc.CANONICAL_VERSION_NAME.la(version));
  }

  public static ItemIdentity generator(long id) {
    return ItemIdentity.longId(CoreItemTypes.GENERATOR, id);
  }
  
  public static ItemIdentity loopMarker(long rowId) {
    return ItemIdentity.longId(CoreItemTypes.LOOP_MARKER, rowId);
  }

  public static ItemIdentity component(ProjectComponent component) {
    return ItemIdentity.longId(CoreItemTypes.COMPONENT, component.getId());
  }

  public static ItemIdentity component(long id) {
    return ItemIdentity.longId(CoreItemTypes.COMPONENT, id);
  }

  public static ItemIdentity option(Option option) {
    Long optionId = option.getOptionId();
    return optionId == null ? null : ItemIdentity.longId(CoreItemTypes.CF_OPTION, optionId);
  }

  public static ItemIdentity option(long optionId) {
    return ItemIdentity.longId(CoreItemTypes.CF_OPTION, optionId);
  }

  public static ItemIdentity structure(long structureId) {
    return ItemIdentity.longId(CoreItemTypes.STRUCTURE, structureId);
  }

  public static ItemIdentity sprint(long sprintId) {
    return ItemIdentity.longId(CoreItemTypes.SPRINT, sprintId);
  }

  public static ItemIdentity tempoAccount(int accountId) {
    return ItemIdentity.longId(CoreItemTypes.TEMPO_ACCOUNT, accountId);
  }

  public static ItemIdentity i18nFolder(@NotNull String i18nKey) {
    return ItemIdentity.stringId(CoreItemTypes.FOLDER, i18nKey);
  }

  public static ItemIdentity textFolder(@NotNull String text) {
    return ItemIdentity.stringId(CoreItemTypes.FOLDER, "/" + text);
  }

  public static ItemIdentity folder(long folderId) {
    return ItemIdentity.longId(CoreItemTypes.FOLDER, folderId);
  }

  public static ItemIdentity sdRequestType(int requestTypeId) {
    return ItemIdentity.longId(CoreItemTypes.REQUEST_TYPE, requestTypeId);
  }

  public static boolean isIssue(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.ISSUE);
  }

  public static boolean isProject(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.PROJECT);
  }

  public static boolean isComponent(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.COMPONENT);
  }

  public static boolean isVersion(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.VERSION);
  }

  public static boolean isGenerator(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.GENERATOR);
  }

  public static boolean isStructure(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.STRUCTURE);
  }

  public static boolean isLoopMarker(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.LOOP_MARKER);
  }

  public static boolean isTempoAccount(ItemIdentity itemId) {
    return isLongType(itemId, CoreItemTypes.TEMPO_ACCOUNT);
  }

  public static boolean isVersionName(ItemIdentity itemId) {
    return isStringType(itemId, CoreItemTypes.VERSION_NAME);
  }

  public static String canonicalVersionName(String versionName) {
    return versionName == null ? null : versionName.trim().toLowerCase(Locale.ROOT);
  }

  private static boolean isStringType(@Nullable ItemIdentity itemId, @NotNull String itemType) {
    return itemId != null && itemType.equals(itemId.getItemType()) && itemId.isStringId();
  }

  private static boolean isLongType(@Nullable ItemIdentity itemId, @NotNull String itemType) {
    return itemId != null && itemType.equals(itemId.getItemType()) && itemId.isLongId();
  }
}
