package com.almworks.jira.structure.api.job;

import com.almworks.jira.structure.api.util.CallableE;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.Future;

/**
 * <p><code>StructureJobManager</code> allows running and scheduling background jobs.</p>
 *
 * <p>The jobs are run in a separate special thread, maintained by the job manager.
 * There may be one thread or several threads that run the jobs.</p>
 *
 * <p>Each time the job is run, the manager sets up the thread environment and thread locals
 * so that the job may use all standard JIRA services. The current user is set to
 * {@link StructureJob#getUser()}.</p>
 *
 * <p>The finished jobs and job feedback are kept in memory.</p>
 *
 * todo describe it is not JDC-aware
 *
 * @author Igor Sereda
 */
public interface StructureJobManager {
  /**
   * A constant representing the user executor.
   * @see #execute(StructureJob, String)
   */
  String USER_EXECUTOR_ID = "user";

  /**
   * A constant representing the system executor.
   * @see #execute(StructureJob, String)
   */
  String SYSTEM_EXECUTOR_ID = "system";

  String GENERATOR_EXECUTOR_ID = "generator";

  String ATTRIBUTE_EXECUTOR_ID = "attribute";

  /**
   * <p>Schedule the job and run it as soon as possible.</p>
   *
   * <p>The job is added to the end of the queue.</p>
   *
   * @param job the job to run
   * @return the ID of the job
   * @throws StructureJobException when the job cannot be scheduled - for example if the component is
   * shut down or if the job has incorrect status
   */
  long enqueue(@NotNull StructureJob job) throws StructureJobException;

  /**
   * <p>Schedule the job and run it as soon as possible.</p>
   *
   * <p>The job bypasses the queue and is executed by the "user" executor.</p>
   *
   * @param job the job to run
   * @return the ID of the job
   * @throws StructureJobException when the job cannot be scheduled - for example if the component is
   * shut down or if the job has incorrect status
   * @since 7.2.0 (Structure 2.0)
   * @see #execute(StructureJob, String)
   * @see #USER_EXECUTOR_ID
   */
  long execute(@NotNull StructureJob job) throws StructureJobException;

  /**
   * <p>Schedule the job and run it as soon as possible.</p>
   *
   * <p>The job bypasses the queue and is executed by a separate executor identified by {@code executorId}.</p>
   *
   * @param job the job to run
   * @param executorId the executor ID
   * @return the ID of the job
   * @throws StructureJobException when the job cannot be scheduled - for example if the component is
   * shut down or if the job has incorrect status; when the executor ID is invalid
   * @since 7.4.0 (Structure 2.3)
   * @see #SYSTEM_EXECUTOR_ID
   * @see #USER_EXECUTOR_ID
   */
  long execute(@NotNull StructureJob job, String executorId) throws StructureJobException;

  /**
   * <p>Schedule the job to run periodically, or once after delay.</p>
   *
   * <p>This method allows to run periodic jobs. Due to the nature of the manager, the intervals
   * are not precise, the job is actually run as soon as possible after the time to run has come.</p>
   *
   * @param initial the initial delay before running the job for the first time, in milliseconds; <code>0</code> means run now
   * @param interval the interval between subsequent runnings of the job, in milliseconds; if <code>0</code>, don't run the job more than once
   * @param job the job to run
   * @return the ID of the job
   * @throws StructureJobException when the job cannot be scheduled - for example if the component is
   * shut down or if the job has incorrect status
   */
  long schedule(long initial, long interval, @NotNull StructureJob job) throws StructureJobException;

  /**
   * Retrieves the job by its ID.
   *
   * @param jobId the id of the job
   * @return the job with the specified ID, or null if no such job exists
   */
  @Nullable
  StructureJob getJob(Long jobId);

  /**
   * Cancels the execution of the job, if possible.
   *
   * @param jobId the ID of the job
   */
  void cancel(Long jobId);

  long execute(@NotNull CallableE<?, ?> callable, @NotNull String executorId) throws StructureJobException;

  @Nullable
  <T> Future<T> getFuture(@Nullable Long jobId);
}
