package com.almworks.jira.structure.api.row;

import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>{@code StructureRow} instances represent rows, main building blocks of a forest.</p>
 *
 * <p>Structures and query results emit forests. {@link Forest} content is expressed as a sequence of row IDs.
 * To get more information about the row, you need to get {@code StructureRow} instance from {@link RowManager}.</p>
 *
 * <p>A row has the following properties:</p>
 *
 * <ul>
 *   <li>{@code rowID} is the unique ID of the row.</li>
 *   <li>{@code itemID} is the ID of the item that is displayed in this row.</li>
 *   <li>{@code semantics} is the semantic code of the relationship between the row and its parent. <em>Currently, semantics are
 *   not supported and are zero most of the times.</em></li>
 *   <li>{@code originalId} and {@code creatorId} are additional properties, specified only for transient rows
 *   &mdash; see {@link RowManager}.</li>
 * </ul>
 *
 * <p>Note that a single forest may contain the same item in multiple places, but all rows will be unique.</p>
 *
 * <p>Rows are immutable. Persistent rows get stored in the database and are never updated nor deleted.
 * Temporary rows get cleared when JIRA instance restarts. <em>Garbage collection for rows is planned for the future.</em></p>
 *
 * <p><strong>Important:</strong> {@code StructureRow} is a temporary object and must not be cached or leaked outside the current request, because it
 * stores the result of access check to the item for the current user.</p>
 *
 */
@PublicApi
public interface StructureRow {
  StructureRow ROW_ZERO = new ZeroRow();

  /**
   * Returns row ID, a positive number.
   */
  long getRowId();

  /**
   * Returns item ID for the item shown in this row.
   */
  @NotNull
  ItemIdentity getItemId();

  /**
   * Returns row's semantics.
   */
  long getSemantics();

  /**
   * <p>Tries to resolve the item based on its ID, that is, get an object that represents this item.
   * Resolution includes checking the current user's access to that item.</p>
   *
   * <p>This method may return {@code null} in many different cases:</p>
   * <ul>
   *   <li>The item does not exist.</li>
   *   <li>The item is not accessible by the current user.</li>
   *   <li>The plugin module that is responsible for this type of items is disabled or missing.</li>
   *   <li>Item's object type is not reducible to {@code itemClass}.</li>
   * </ul>
   *
   * @param itemClass expected class of the item's object
   * @param <I> type of item's objects
   * @return the item's object, or {@code null}
   */
  @Nullable
  <I> I getItem(@NotNull Class<I> itemClass);


  final class ZeroRow implements StructureRow {
    @Override
    public long getRowId() {
      return 0;
    }

    @NotNull
    @Override
    public ItemIdentity getItemId() {
      return ItemIdentity.ITEM_ZERO;
    }

    @Nullable
    @Override
    public <I> I getItem(@NotNull Class<I> itemClass) {
      return null;
    }

    @Override
    public long getSemantics() {
      return 0;
    }

    @Override
    public String toString() {
      return getRowId() + ":" + getItemId();
    }
  }
}
