package com.almworks.jira.structure.api.sync;

import com.almworks.jira.structure.api.error.StructureErrors;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.util.StructureUtil;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.xml.bind.*;
import java.io.ByteArrayInputStream;
import java.util.Map;

public interface Structure2xBackwardCompatibleSynchronizer extends StructureSynchronizer {
  /**
   * @return null if parameters migration fails
   */
  @Nullable
  Map<String, Object> migrateParameters(@NotNull byte[] params) throws StructureException;

  class SyncMigrationUtil {
    private static final Logger log = LoggerFactory.getLogger(SyncMigrationUtil.class);
  
    
    @NotNull
    public static <P> P restoreStructure2xParameters(byte[] data, Class<P> paramsClazz) throws StructureException {
      // Important! - this is called during restore(), not during synchronization, so we cannot use SyncLogger.get() here.
      
      if (data == null) {
        throw StructureErrors.INVALID_PARAMETER.withMessage("missing parameters");
      }
      try {
        JAXBContext jaxbContext = StructureUtil.createJAXBContext(paramsClazz);
        if (jaxbContext == null) {
          log.warn("cannot create JAXB context for " + paramsClazz);
          throw StructureErrors.INVALID_PARAMETER.withMessage("cannot migrate parameters");
        }
        Unmarshaller unmarshaller = jaxbContext.createUnmarshaller();
        Object object = unmarshaller.unmarshal(new ByteArrayInputStream(data));
        if (object == null) {
          throw StructureErrors.INVALID_PARAMETER.withMessage("empty parameters after migration");
        }
        if (!paramsClazz.isInstance(object)) {
          log.warn("cannot unmarshal synchronizer parameters, unexpected class " + object.getClass());
          throw StructureErrors.INVALID_PARAMETER.withMessage("incompatible parameters class " + object.getClass());
        }
        return paramsClazz.cast(object);
      } catch (JAXBException e) {
        log.error("error unmarshalling parameters", e);
        throw StructureErrors.INVALID_PARAMETER.causedBy(e).withMessage("cannot migrate parameters");
      }
    }
  }
}
