package com.almworks.jira.structure.api.sync.util;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.StructurePluginHelper;
import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.*;
import com.almworks.jira.structure.api.forest.ForestSource;
import com.almworks.jira.structure.api.forest.action.*;
import com.almworks.jira.structure.api.forest.item.ItemForest;
import com.almworks.jira.structure.api.forest.item.ItemForestBuffer;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.item.CoreIdentities;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.row.RowManager;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.jira.bc.JiraServiceContextImpl;
import com.atlassian.jira.bc.filter.SearchRequestService;
import com.atlassian.jira.issue.search.SearchRequest;
import com.atlassian.jira.jql.parser.*;
import com.atlassian.jira.util.ErrorCollection;
import com.atlassian.jira.util.MessageSet;
import com.atlassian.jira.web.action.JiraWebActionSupport;
import com.atlassian.query.Query;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.almworks.jira.structure.api.error.StructureErrors.*;
import static com.almworks.jira.structure.api.util.StructureUtil.nnv;

public class SyncUtil {
  public static final SyncChangeListener DUMMY_LISTENER =  new SyncChangeListener() {
    @Override
    public void onInsert(long parentRow, LongIterable addedIssues) {
      // do nothing
    }

    @Override
    public void onMove(long parentRow, long issueId) {
      // do nothing
    }
  };

  /**
   * <p>Merges the subtree of {@code row} under {@code targetRow}.
   * See Synchronizers page on the wiki for the description of the merge-move operation.</p>
   * 
   * <p>In brief, this a recursive procedure: it ensures that all items in the subtree of {@code row} are present in the subtree
   * of {@code targetRow}, preserving the hierarchy; it is achieved by either moving rows for items that are not present 
   * or recursively applying the same procedure to the rows for items that are present.</p>
   * 
   * <p>If any move fails because some items are inaccessible or because parent item permissions do not allow it,
   * the rows affected by that move and all their ancestors remain in place. Otherwise, {@code row} is removed from the forest.</p>
   *
   * <p>Most commonly, one would need to use the simpler {@link #merge(ForestSource, RowManager, long, long)} override.</p>
   * 
   * @param ufs does the necessary changes
   * @param rowManager row manager
   * @param row the "donor" row: its subtree will be merged/moved into the subtree of the target row, and it will be removed.
   *   The exception is when some move fails, then the row will stay in place along with the rows that failed to move
   *   and their ancestors.
   * @param targetRow target row - the "receiver" row; 0 means that donor row children will be merged into roots recursively
   * @param followSymlinks if true, when merging subtrees will consider cycle markers equal with the items they represent
   * @param removeRow whether the "donor" row should be removed. Is not recursive: children of donor row that are merged
   *                  will be removed regardless of this setting
   * @param mergedRows if not null, all merges will be recorded here: when a row is removed because a row for the same item
   *                   exists under the target row, a mapping removedRow -> rowForSameItem
   * 
   * @return {@code true} if the merge has started; that means that some rows might have been moved, some might have been deleted,
   *         but it's also possible that nothing has happened. 
   *         {@code false} if the merge hasn't started because some of the preconditions are not met: merged row is ancestor of target row, or 
   *              either merged row or target row is missing (target row 0 is considered to be "present").
   * 
   * @throws StructureException if any unrecoverable problem happens, such as: inaccessible structure, unlicensed usage,
   *   an internal error, too many concurrent changes, etc.
   * */
  public static boolean merge(ForestSource ufs, RowManager rowManager, long row, long targetRow,
    boolean followSymlinks, boolean removeRow, @Nullable WritableLongLongMap mergedRows)
    throws StructureException
  {
    if (row == targetRow) {
      throw StructureErrors.INVALID_FOREST_OPERATION.forRow(row).withMessage("cannot merge row " + row + " into itself");
    }
    
    Forest forest = ufs.getLatest().getForest();
    int rowIdx = forest.indexOf(row);
    int targetRowIdx = targetRow == 0 ? -1 : forest.indexOf(targetRow);
    
    if (rowIdx < 0 || (targetRow != 0 && targetRowIdx < 0)) {
      SyncLogger slog = SyncLogger.get();
      if (slog.getLogger().isDebugEnabled()) {
        slog.debug("skipping merge of", slog.row(row), "into", slog.row(targetRow), ": one of them is not in the current forest");
      }
      return false;
    }
    if (targetRowIdx >= 0 && subtreeContains(forest, rowIdx, targetRowIdx)) {
      SyncLogger slog = SyncLogger.get();
      if (slog.isInfoEnabled()) {
        slog.info("skipping merge of", slog.row(row), "into", slog.row(targetRow), ": target row", targetRow,
          "is in the subtree of donor row", row);
      }
      return false;
    }

    SyncLogger slogDebug = null;
    if (SyncLogger.isDebug()) {
      slogDebug = SyncLogger.get();
      slogDebug.debug("_merge_", slogDebug.row(row), "into", slogDebug.row(targetRow));
      slogDebug.pushPrefix("merge");
    }
    try {
      merge0(ufs, forest, rowManager, row, rowIdx, targetRowIdx, rowIdx, followSymlinks, removeRow, mergedRows, new boolean[]{true});
    } finally {
      if (slogDebug != null) slogDebug.popPrefix();
    }
    
    return true;
  }

  public static boolean merge(ForestSource ufs, RowManager rowManager, long row, long targetRow)
    throws StructureException
  {
    return merge(ufs, rowManager, row, targetRow, false, true, null);
  }

  public static boolean merge(ForestSource ufs, RowManager rowManager, long row, long targetRow, boolean followSymlinks)
    throws StructureException
  {
    return merge(ufs, rowManager, row, targetRow, followSymlinks, true, null);
  }

  private static boolean subtreeContains(Forest forest, int idx, int checkIdx) {
    for (int size = forest.size(), startDepth = forest.getDepth(idx++); idx < size && forest.getDepth(idx) > startDepth; ++idx) {
      if (idx == checkIdx) {
        return true;
      }
    }
    return false;
  }

  // row, rowIdx: donor row
  // targetRowIdx: target row
  // origDonorIdx: the topmost donor row for which merge() was called - to prevent from using it as a target row
  private static void merge0(ForestSource ufs, Forest forest, RowManager rowManager,
    long row, int rowIdx, int targetRowIdx, int origDonorIdx,
    boolean followSymlinks, boolean removeRow, WritableLongLongMap mergedRows,
    boolean[] subtreeMergeSuccess
  )
    throws StructureException
  {
    long targetRow = targetRowIdx < 0 ? 0 : forest.getRow(targetRowIdx);
    IntIterator allTargetChildrenIdxIt = forest.getChildrenIndicesIterator(targetRowIdx);
    IntIterator targetChildrenIdxIt = allTargetChildrenIdxIt;
    if (targetChildrenIdxIt.hasNext()) {
      // Ignore origDonor as if it's not in the forest anymore
      targetChildrenIdxIt = new IntMinusIterator(allTargetChildrenIdxIt, new IntIterator.Single(origDonorIdx));
    }
    Map<ItemIdentity, Integer> targetChildrenItemIdIndex = buildItemIdIndex(forest, rowManager, followSymlinks, targetChildrenIdxIt);
    // Even though donor row cannot be a "target child", it still should be used as "after"
    long after = allTargetChildrenIdxIt.hasValue() ? forest.getRow(allTargetChildrenIdxIt.value()) : 0;

    LongArray childrenToMove = new LongArray();
    for (IntIterator cii : forest.getChildrenIndicesIterator(rowIdx)) {
      int childRowIdx = cii.value();
      long childRow = forest.getRow(childRowIdx);
      ItemIdentity childItemId = getItemId(rowManager, childRow, followSymlinks);
      Integer targetChildIdx = targetChildrenItemIdIndex.get(childItemId);
      if (targetChildIdx != null) {
        // Move the block of children rows not present under target row
        if (!childrenToMove.isEmpty()) {
          after = move(ufs, childrenToMove, targetRow, after, subtreeMergeSuccess);
          childrenToMove.clear();
        }
        // Merge subtree
        merge0(ufs, forest, rowManager, childRow, childRowIdx, targetChildIdx, origDonorIdx, followSymlinks, true,
          mergedRows, subtreeMergeSuccess);
      } else {
        childrenToMove.add(childRow);
      }
    }

    // Move the remaining block of children rows not present under target row
    if (!childrenToMove.isEmpty()) {
      after = move(ufs, childrenToMove, targetRow, after, subtreeMergeSuccess);
    }

    if (removeRow && subtreeMergeSuccess[0]) {
      // todo collect rows to remove and bulk remove
      remove(ufs, new LongList.Single(row));
      if (mergedRows != null) {
        mergedRows.put(row, targetRow);
      }
    }
  }

  /**
   * @param success if move was successful, is not changed; if move wasn't successful, success[0] == false 
   * */
  public static long move(ForestSource ufs, LongList rows, long under, long after, boolean[] success)
    throws StructureException
  {
    return move(ufs, rows, under, after, false, success);
  }

  /**
   * @param success if move was successful, is not changed; if move wasn't successful, success[0] == false 
   * */
  public static long move(ForestSource ufs, @Nullable LongList rows, long under, long after, boolean invalidMoveExpected,
    boolean[] success)
    throws StructureException
  {
    if (rows == null || rows.isEmpty()) return after;
    ForestAction.Move move = new ForestAction.Move(rows, under, after, 0);
    // If some of the rows that we are about to move disappear from the forest, we don't care and we don't want to break the synchronization process because of that
    try {
      ufs.apply(move, ActionParameters.IGNORE_MOVED_ROW_MISSING_MAP);
      if (SyncLogger.isDebug()) {
        SyncLogger slog = SyncLogger.get();
        slog.debug("_move_", slog.rows(rows), "under:", slog.row(under), "after:", slog.row(after));
      }
      return rows.get(rows.size() - 1);
    } catch (StructureException e) {
      processMoveError(e, under, rows, after, invalidMoveExpected);
      if (success != null) {
        success[0] = false;
      }
      return after;
    }
  }

  private static void processMoveError(StructureException exception, long targetRow, LongList rows, long after,
    boolean invalidMoveExpected) 
    throws StructureException 
  {
    SyncLogger slog = SyncLogger.get();
    StringBuilder errorMessage = new StringBuilder("could not move ");
    slog.appendRows(rows, errorMessage)
      .append(" under ")
      .append(slog.row(targetRow))
      .append(" after ")
      .append(slog.row(after))
      .append(": ");

    if (invalidMoveExpected && exception.getError().equals(StructureErrors.INVALID_FOREST_OPERATION) && slog.getLogger().isInfoEnabled()) {
      slog.info(errorMessage.
        append(exception.getProblemDetails()).append(". ").
        append("Most likely, it is caused by previous changes made by this synchronizer; in that case, it is not a problem. ").
        append("Another possible but unlikely reason: concurrent changes made by users. If it can be the case, please run a resync."));
    } else {
      appendStructureException(exception, errorMessage);
      slog.warn(errorMessage.toString());
    }
  }

  public static void remove(ForestSource ufs, @Nullable LongList rows) throws StructureException {
    if (rows == null || rows.isEmpty()) return;
    ForestAction.Remove remove = new ForestAction.Remove(rows);
    try {
      ufs.apply(remove, ActionParameters.IGNORE_REMOVED_ROW_MISSING_MAP);
      if (SyncLogger.isDebug()) {
        SyncLogger.get().debug("_remove_", SyncLogger.get().rows(rows));
      }
    } catch (StructureException e) {
      SyncLogger slog = SyncLogger.get();
      StringBuilder errorMessage = new StringBuilder("could not remove ");
      slog.appendRows(rows, errorMessage).append(": ");
      processUpdateError(e, slog, errorMessage);
    }
  }
  
  /**
   * Returns row ID replacements, if the add was successful; returns {@code null} otherwise.
   * */
  @Nullable
  public static LongLongMap add(ForestSource ufs, ItemForest fragment, long under, long after) throws StructureException {
    ForestAction.Add add = new ForestAction.Add(fragment, under, after, 0);
    return add(ufs, add);
  }

  @Nullable
  public static LongLongMap add(ForestSource ufs, ForestAction.Add add) throws StructureException {
    try {
      ActionResult actionResult = ufs.apply(add, ActionParameters.IGNORE_ADD_AFTER_ROW_PROBLEMS_MAP);
      LongLongMap rowIdReplacements = actionResult.getRowIdReplacements();
      if (SyncLogger.isDebug()) {
        SyncLogger slog = SyncLogger.get();
        slog.debug("_add_", slog.itemForest(add.getFragment()), "under:", slog.row(add.getUnder()),
          "after:", slog.row(add.getAfter()), "before:", slog.row(add.getBefore()),
          "replacements:", rowIdReplacements);
      }
      return rowIdReplacements;
    } catch (StructureException e) {
      boolean before = add.getBefore() != 0;
      SyncLogger slog = SyncLogger.get();
      StringBuilder errorMessage = new StringBuilder("could not add ");
      slog.appendItemForest(add.getFragment(), errorMessage)
        .append(" under ")
        .append(slog.row(add.getUnder()))
        .append(before ? " before " : " after ")
        .append(slog.row(before ? add.getBefore() : add.getAfter()))
        .append(": ");
      processUpdateError(e, slog, errorMessage);
      return null;
    }
  }

  public static void insert(ForestSource ufs, ItemForestBuffer addedForest, LongList issues,
    long under, long after) throws StructureException
  {
    addedForest.clear();
    for (LongIterator it: issues) {
      addedForest.add(CoreIdentities.issue(it.value()));
    }
    add(ufs, addedForest, under, after);
  }

  public static void processUpdateError(StructureException exception, SyncLogger slog, StringBuilder errorMessage)
    throws StructureException
  {
    appendStructureException(exception, errorMessage);
    slog.warn(errorMessage.toString());
  }

  private static StringBuilder appendStructureException(StructureException se, StringBuilder errorMessage)
    throws StructureException
  {
    StructureError error = se.getError();
    if (error.is(StructureErrorCategory.NOT_FOUND)) {
      errorMessage.append("could not access item ").append(se.getItem());
      return errorMessage;
    }
    if (error == INVALID_FOREST_OPERATION) {
      // INVALID_FOREST_OPERATION - caused by update sanity checker finding out that target coordinates are no longer in the structure - indicates concurrent change
      errorMessage.
        append(se.getProblemDetails()).append(se.getProblemDetails().endsWith(".") ? " " : ". ").
        append("This might be a temporary problem caused by concurrent changes to this structure. ").
        append("To remedy the situation, try running resync. If the problem persists, please contact support@almworks.com.");
    } else if (error.getCode() == 6014) {
      // See InternalErrors.CONCURRENT_UPDATE_FAILED
      errorMessage.
        append(se.getProblemDetails()).append(". ").
        append(" To remedy the situation, try running resync. If the problem persists, please contact support@almworks.com.");
    } else if (error.isOneOf(UNAVAILABLE_MODULE, FOREST_CHANGE_PROHIBITED_BY_PARENT_PERMISSIONS)) {
      errorMessage.append(se.getProblemDetails());
    } else {
      // All other errors are considered critical, all further sync operations will be aborted
      throw se;
    }
    return errorMessage;
  }

  private static Map<ItemIdentity, Integer> buildItemIdIndex(Forest forest, RowManager rowManager,
    boolean followSymlinks, IntIterator idxs)
  {
    if (!idxs.hasNext()) return Collections.emptyMap();
    Map<ItemIdentity, Integer> firstRowIdxs = new HashMap<>();
    for (IntIterator i : idxs) {
      ItemIdentity itemId = getItemId(rowManager, forest.getRow(i.value()), followSymlinks);
      if (!firstRowIdxs.containsKey(itemId)) {
        firstRowIdxs.put(itemId, i.value());
      }
    }
    return firstRowIdxs;
  }
  
  /**
   * @return true if parent was moved out of its child subtree
   * @throws StructureException
   */
  public static boolean resolveMoveIntoOwnSubtree(long moveTo, long row, HierarchyHelper hh,
    ForestSource ufs, @Nullable SyncLogger debugLog) throws StructureException
  {
    if (hh.isInSubtree(moveTo, row)) {
      long parentRow = hh.getParent(row);
      if (debugLog != null) debugLog.debug("moving parent", debugLog.row(moveTo), "out of child subtree", debugLog.row(row));
      move(ufs, new LongList.Single(moveTo), parentRow, row, null);
      hh.recordMove(moveTo, parentRow);
      return true;
    }
    return false;
  }

  @NotNull
  private static ItemIdentity getItemId(RowManager rowManager, long row, boolean followSymlinks) {
    ItemIdentity itemId = rowManager.getRow(row).getItemId();
    if (followSymlinks) {
      while (CoreIdentities.isLoopMarker(itemId)) {
        itemId = rowManager.getRow(itemId.getLongId()).getItemId();
      }
    }
    return itemId;
  }

  public static long getFilterId(@NotNull Map<String, ?> params, @NotNull String name, SearchRequestService searchService,
    JiraWebActionSupport action) 
  {
    String sf = StructureUtil.getSingleParameter(params, name);
    if (sf != null) {
      Pattern filterPattern = Pattern.compile("filter-(\\d+)");
      Matcher m = filterPattern.matcher(sf);
      if (m.matches()) {
        long filterId = StructureUtil.lv(m.group(1), 0);
        if (filterId > 0) {
          JiraServiceContextImpl context = new JiraServiceContextImpl(StructureAuth.getUser());
          SearchRequest filter = searchService.getFilter(context, filterId);
          ErrorCollection ec = context.getErrorCollection();
          if (ec.hasAnyErrors()) {
            action.addError(name, StringUtils.join(ec.getErrorMessages(), "\n"));
            return 0;
          }
          if (filter == null) {
            action.addError(name, action.getText("s.sync.error.bad-filter"));
          }
          return filterId;
        }
      }
    }
    action.addError(name, action.getText("s.sync.error.no-filter-id"));
    return 0;
  }

  @Nullable
  public static String getJqlQuery(@NotNull Map<String, ?> params, @NotNull String name, StructurePluginHelper helper,
    JiraWebActionSupport action) 
  {
    String jql = StructureUtil.getSingleParameter(params, name);
    if (StringUtils.isBlank(jql)) {
      action.addError(name, action.getText("s.sync.error.no-jql-query"));
      return null;
    }
    jql = jql.trim();
    Query query;
    try {
      query = helper.getJqlQueryParser().parseQuery(jql);
    } catch (JqlParseException e) {
      JqlParseErrorMessage jpem = nnv(e.getParseErrorMessage(), JqlParseErrorMessages.genericParseError());
      action.addError(name, jpem.getLocalizedErrorMessage(helper.getI18n()));
      return null;
    }
    MessageSet errors = helper.validateQuery(StructureAuth.getUser(), query);
    if (errors.hasAnyErrors()) {
      action.addError(name, StringUtils.join(errors.getErrorMessages(), "\n"));
      return null;
    }
    return jql;
  }
}
