package com.almworks.jira.structure.api.util;

import com.atlassian.annotations.PublicApi;
import com.google.common.base.Function;
import org.jetbrains.annotations.Nullable;

/**
 * Helper methods for using numeric functions.
 */
@PublicApi
public final class NumericFunctions {
  /**
   * @param number a number
   * @return The given number's {@code long} value, 0L if {@code null}
   */
  public static long longOrZero(Number number) {
    return number == null ? 0L : number.longValue();
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return The given function's {@code long} value for the argument, 0L if anything is {@code null}
   */
  public static <T> long longOrZero(T argument, Function<? super T, ? extends Number> function) {
    if (argument == null || function == null) {
      return 0L;
    }
    return longOrZero(function.apply(argument));
  }

  /**
   * @param number a number
   * @return A {@code Long} with the given number's {@code long} value, {@code null} if {@code number} is {@code null}
   */
  @Nullable
  public static Long longOrNull(Number number) {
    return number == null ? null : number instanceof Long ? (Long) number : Long.valueOf(number.longValue());
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return A {@code Long} with the given function's {@code long} value for the argument, {@code null} if anything is {@code null}
   */
  @Nullable
  public static <T> Long longOrNull(T argument, Function<? super T, ? extends Number> function) {
    if (argument == null || function == null) {
      return null;
    }
    return longOrNull(function.apply(argument));
  }

  /**
   * @param number a number
   * @return The given number's {@code long} value, 0L if {@code null} or negative
   */
  public static long nonNegativeLong(Number number) {
    return Math.max(0L, longOrZero(number));
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return The given function's {@code long} value for the argument, 0L if anything is {@code null} or negative
   */
  public static <T> long nonNegativeLong(T argument, Function<? super T, ? extends Number> function) {
    return Math.max(0L, longOrZero(argument, function));
  }

  /**
   * @param number a number
   * @return The given number's {@code double} value, 0.0 if {@code null}
   */
  public static double doubleOrZero(Number number) {
    return number == null ? 0.0 : number.doubleValue();
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return The given function's {@code double} value for the argument, 0.0 if anything is {@code null}
   */
  public static <T> double doubleOrZero(T argument, Function<? super T, ? extends Number> function) {
    if (argument == null || function == null) {
      return 0.0;
    }
    return doubleOrZero(function.apply(argument));
  }

  /**
   * @param number a number
   * @return A {@code Double} with the given number's {@code double} value, {@code null} if {@code number} is {@code null}
   */
  @Nullable
  public static Double doubleOrNull(Number number) {
    return number == null ? null : number instanceof Double ? (Double) number : Double.valueOf(number.doubleValue());
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return A {@code Double} with the given function's {@code double} value for the argument, {@code null} if anything is {@code null}
   */
  @Nullable
  public static <T> Double doubleOrNull(T argument, Function<? super T, ? extends Number> function) {
    if (argument == null || function == null) {
      return null;
    }
    return doubleOrNull(function.apply(argument));
  }

  /**
   * @param number a number
   * @return The given number's {@code double} value, 0.0 if {@code null} or negative
   */
  public static double nonNegativeDouble(Number number) {
    return Math.max(0.0, doubleOrZero(number));
  }

  /**
   * @param argument an argument
   * @param function a numeric function
   * @return The given function's {@code double} value for the argument, 0.0 if anything is {@code null} or negative
   */
  public static <T> double nonNegativeDouble(T argument, Function<? super T, ? extends Number> function) {
    return Math.max(0.0, doubleOrZero(argument, function));
  }

  /**
   * Appends two numbers as double's with tracking of nulls and clipping to zero.
   * Axioms: null + a == a, a + null == a, a + b == a + b, null + null == null
   *
   * @param a first operand
   * @param b second operand
   * @return a + b or null
   */
  public static Double nonNegativeDoubleAddOrNull(Number a, Number b) {
    if(a == null && b == null)
      return null;
    return nonNegativeDouble(a) + nonNegativeDouble(b);
  }

  /**
   * Appends two numbers as long's with tracking of nulls
   * Axioms: null + a == a, a + null == a, a + b == a + b, null + null == null
   *
   * @param a first operand
   * @param b second operand
   * @return a + b or null
   */
  public static Long longAddOrNull(Number a, Number b) {
    if(a == null && b == null)
      return null;
    if(a == null)
      return b.longValue();
    if(b == null)
      return a.longValue();
    return a.longValue() + b.longValue();
  }

  public static long longAddOrNull(long a, Long b) {
    if (b == null) return a;
    return a + b;
  }

  public static boolean equals(Number a, Number b) {
    if (a == null) {
      return b == null;
    }
    return a.equals(b);
  }
}
