package com.almworks.jira.structure.api.effector.process;

import com.almworks.jira.structure.api.effector.Effector;
import com.almworks.jira.structure.api.effector.instance.EffectorInstance;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.*;

/**
 * Manages {@link Effector} preview, effect application, and undo background
 * processes.
 */
@PublicApi
public interface EffectorProcessManager {
  /**
   * Checks whether an effector process can be started, as if by calling
   * {@link #startProcess(List, long, boolean)}, without actually starting it.
   * Returns normally if such a process can be started, throws a
   * {@link StructureException} otherwise.
   *
   * @param effectorInstances effector instances to run, the scope is defined by
   *        the structure ID and position
   * @param structureId ID of the structure for which effects are calculated.
   * @throws StructureException if the process cannot be started, e.g. due to
   *         missing permissions or other problems.
   */
  void validateStartProcess(@NotNull List<EffectorInstance> effectorInstances, long structureId) throws StructureException;

  /**
   * Checks whether an effector process can be started, as if by calling
   * {@link #startProcess(String, Map, ForestSpec, boolean)}, without actually starting it.
   * Returns normally if such a process can be started, throws a
   * {@link StructureException} otherwise.
   *
   * @param moduleKey complete module key of the effector
   * @param parameters effector parameters
   * @param forestSpec forest spec, the scope will be the whole forest
   * @throws StructureException if the process cannot be started, e.g. due to
   *         missing permissions or other problems.
   */
  void validateStartProcess(@NotNull String moduleKey, @NotNull Map<String, Object> parameters,
    @NotNull ForestSpec forestSpec) throws StructureException;

  /**
   * Starts a new effector process for installed effector instances. The current
   * user will be the process owner.
   *
   * @param effectorInstances effector instances to run, the scope is defined by
   *        the structure id and position
   * @param structureId id of the structure for which effects are calculated.
   * @param requiresConfirmation if {@code false}, don't wait for confirmation, start applying
   *        effects as soon as the preview is ready (if successful).
   * @return id of the new process
   * @throws StructureException if the process cannot be started, e.g. due to
   *         missing permissions or other problems.
   */
  long startProcess(
    @NotNull List<EffectorInstance> effectorInstances, long structureId, boolean requiresConfirmation)
    throws StructureException;

  /**
   * Starts a one-off effector process. The current user will be the process
   * owner.
   *
   * @param moduleKey complete module key of the effector
   * @param parameters effector parameters
   * @param forestSpec forest spec, the scope will be the whole forest
   * @param requiresConfirmation if {@code false}, don't wait for confirmation, start applying
   *        effects as soon as the preview is ready (if successful).
   * @return id of the new process
   * @throws StructureException missing permissions, no such effector module,
   *         missing or invalid parameters, invalid forest spec, other problems
   */
  long startProcess(
    @NotNull String moduleKey, @NotNull Map<String, Object> parameters,
    @NotNull ForestSpec forestSpec, boolean requiresConfirmation) throws StructureException;

  /**
   * <p>Starts a new process which undoes the effects performed by an earlier
   * process. The process being undone must be in
   * {@link EffectorProcess.Status#COMPLETED COMPLETED},
   * {@link EffectorProcess.Status#COMPLETED_WITH_ERRORS COMPLETED_WITH_ERRORS},
   * {@link EffectorProcess.Status#APPLY_INTERRUPTED APPLY_INTERRUPTED},
   * or {@link EffectorProcess.Status#APPLY_STOPPED APPLY_STOPPED} status.
   *
   * <p>The current user will be the new process owner. This method will throw
   * a {@link StructureException} if the current user is not the owner of the
   * process identified by {@code processId} and not a Jira admin.
   *
   * <p>The new process proceeds to the apply queue immediately.
   *
   * @param processId the process id whose effects are to be undone
   * @param recordIndexes the effect record indexes to be undone, or
   *        {@code null} for all process records
   * @return id of the new process
   * @throws StructureException no access to process, process doesn't exist or
   * is not finished, or its effect records cannot be found
   */
  long startUndoProcess(long processId, @Nullable Collection<Integer> recordIndexes) throws StructureException;

  /**
   * <p>Retrieves the current state of the process.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @return process state
   * @throws StructureException no access to process or process doesn't exist
   */
  @NotNull
  EffectorProcess getProcess(long processId) throws StructureException;

  /**
   * <p>Retrieves the list of effect records that describe the changes made by
   * the given effector process or the errors that occurred while applying
   * effects. Can be non-empty if the process is in one of the following
   * statuses: {@link EffectorProcess.Status#IN_PROGRESS IN_PROGRESS},
   * {@link EffectorProcess.Status#COMPLETED COMPLETED},
   * {@link EffectorProcess.Status#COMPLETED_WITH_ERRORS COMPLETED_WITH_ERRORS},
   * {@link EffectorProcess.Status#APPLY_INTERRUPTED APPLY_INTERRUPTED},
   * or {@link EffectorProcess.Status#APPLY_STOPPED APPLY_STOPPED}.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @return a list of effect records
   * @throws StructureException no access to process or process doesn't exist
   */
  @NotNull
  List<EffectRecord> getEffectRecords(long processId) throws StructureException;

  /**
   * <p>Starts applying effects for an existing process with a calculated
   * preview. The process must be in the {@link EffectorProcess.Status#CALCULATED
   * CALCULATED} status.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @throws StructureException no access to process, or it doesn't exist, is not
   * in the {@link EffectorProcess.Status#CALCULATED CALCULATED} status
   */
  void confirm(long processId) throws StructureException;

  /**
   * Deprecated. Use {@link EffectorProcessManager#confirm(long)} instead.
   *
   * @param processId process ID
   * @param effectIndices used to filter effects from the preview.
   * @throws StructureException no access to process, or it doesn't exist, is not
   * in the {@link EffectorProcess.Status#CALCULATED CALCULATED} status, or no
   * effects can be found in the preview for the given indices
   */
  @Deprecated
  void confirm(long processId, @Nullable Collection<Integer> effectIndices) throws StructureException;

  /**
   * <p>Asks a process to stop. The operation is asynchronous, so there is no
   * guarantee that process will be stopped after this call.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @throws StructureException no access to process or process doesn't exist
   */
  void cancel(long processId) throws StructureException;

  /**
   * <p>Sets the "acknowledged" flag for a finished process, so it is no longer
   * shown in the list of current processes. The process must be in the
   * {@link EffectorProcess.Status#COMPLETED COMPLETED},
   * {@link EffectorProcess.Status#COMPLETED_WITH_ERRORS COMPLETED_WITH_ERRORS},
   * {@link EffectorProcess.Status#APPLY_INTERRUPTED APPLY_INTERRUPTED},
   * or {@link EffectorProcess.Status#APPLY_STOPPED APPLY_STOPPED} status.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @throws StructureException no access to process, or process doesn't exist,
   * or it is not in one of the "finished" statuses
   */
  void acknowledgeFinished(long processId) throws StructureException;

  /**
   * <p>Deletes a calculated process so that it is no longer shown in the list
   * of processes. The process must be in the
   * {@link EffectorProcess.Status#CALCULATED CALCULATED},
   * {@link EffectorProcess.Status#CALCULATION_STOPPED CALCULATION_STOPPED},
   * or {@link EffectorProcess.Status#CALCULATION_FAILED CALCULATION_FAILED}
   * status.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @throws StructureException no access to process, or process doesn't exist,
   * or it is not in one of the "calculated" statuses
   */
  void deleteCalculated(long processId) throws StructureException;

  /**
   * <p>Moves an interrupted process back to the apply queue to resume applying
   * previously calculated effects in its preview. The process must be in the
   * {@link EffectorProcess.Status#APPLY_INTERRUPTED APPLY_INTERRUPTED} status.
   *
   * <p>This method will throw a {@link StructureException} if the current user
   * is not the process owner and not a Jira admin.
   *
   * @param processId process ID
   * @throws StructureException no access to process, or process doesn't exist,
   * or it is not in the "interrupted" status
   */
  void resumeInterrupted(long processId) throws StructureException;

  /**
   * <p>Finds effector processes by owner, status, or structure ID. The three
   * criteria are combined by logical AND. Passing {@code null} for any
   * parameter means no restriction.
   *
   * <p>This method does not check the current user's access to the returned
   * processes.
   *
   * @param user process owner, pass {@code null} for all owners
   * @param status process status, pass {@code null} for all statuses
   * @param structureId structure ID, pass {@code null} for all structures
   * @return the list of processes matching the search criteria
   */
  @NotNull
  List<EffectorProcess> findProcesses(@Nullable ApplicationUser user, @Nullable EffectorProcess.Status status, @Nullable Long structureId);

  /**
   * <p>Finds all effector processes for the given user, except finished
   * processes for which the "acknowledged" flag is set.
   *
   * <p>This method does not check the current user's access to the returned
   * processes.
   *
   * @param user the user
   * @return the list of current processes for the given user
   */
  @NotNull
  List<EffectorProcess> getRunningProcessesForUser(@NotNull ApplicationUser user);
}
