package com.almworks.jira.structure.api.forest;

import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.Immutable;

/**
 * {@code VersionedForest} represents a {@link Forest}, tagged with a {@link DataVersion}. It is used by
 * {@link ForestSource} to provide versioned updates about forest content.
 *
 * @see ForestSource
 * @see VersionedForestUpdate
 */
@PublicApi
@Immutable
public final class VersionedForest {
  public static final VersionedForest EMPTY = new VersionedForest(Forest.EMPTY, DataVersion.ZERO);

  @NotNull
  private final Forest myForest;

  @NotNull
  private final DataVersion myVersion;

  /**
   * Constructs an instance.
   *
   * @param forest the forest, must be immutable!
   * @param version version associated with the forest
   */
  public VersionedForest(@NotNull Forest forest, @NotNull DataVersion version) {
    //noinspection ConstantConditions
    if (forest == null) {
      throw new IllegalArgumentException("forest cannot be null");
    }
    //noinspection ConstantConditions
    if (version == null) {
      throw new IllegalArgumentException("version cannot be null");
    }
    assert forest.isImmutable() : forest;
    myForest = forest;
    myVersion = version;
  }

  /**
   * Returns the forest.
   */
  @NotNull
  public Forest getForest() {
    return myForest;
  }

  /**
   * Returns the version.
   */
  @NotNull
  public DataVersion getVersion() {
    return myVersion;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    VersionedForest that = (VersionedForest) o;

    if (!myForest.equals(that.myForest)) return false;
    if (!myVersion.equals(that.myVersion)) return false;

    return true;
  }

  @Override
  public int hashCode() {
    int result = myForest.hashCode();
    result = 31 * result + myVersion.hashCode();
    return result;
  }

  @Override
  public String toString() {
    return "VersionedForest{" + myForest + ", " + myVersion + '}';
  }
}
