package com.almworks.jira.structure.api.generator;

import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.forest.item.*;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.almworks.jira.structure.api.item.ItemVersionUpdate;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.almworks.jira.structure.api.row.StructureRow;
import com.almworks.jira.structure.api.util.RunnableE;
import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.PublicSpi;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.ErrorCollection;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Map;
import java.util.Set;

@PublicSpi
public interface StructureGenerator {
  /**
   * Fills the parameter map for the edit form Velocity template with default
   * values for a new generator item of this type.
   *
   * @param form form template parameters (output)
   * @throws GeneratorUnavailableException if the operation cannot be performed
   */
  void addDefaultFormParameters(@NotNull Map<String, Object> form) throws GeneratorUnavailableException;

  /**
   * Given the generator-specific parameters, fills the parameter map for
   * the edit form Velocity template.
   *
   * @param parameters generator parameters (input)
   * @param form edit form template parameters (output)
   * @throws GeneratorUnavailableException if the operation cannot be performed
   */
  void addParametersToForm(@NotNull Map<String, Object> parameters, @NotNull Map<String, Object> form) throws GeneratorUnavailableException;

  /**
   * <p>Validates the parameters from the edit HTML form and constructs a
   * serializable map of generator-specific parameters.
   *
   * <p>Any errors for invalid input parameters must be reported to the provided
   * {@link ErrorCollection}; if it contains any error messages after this
   * method is called, they are reported to the user and the resulting map
   * is ignored.
   *
   * <p>The resulting parameter map could be stored in the database or
   * transferred over the wire, so it must be serializable to JSON. We advise
   * that you put in only simple objects (strings, numbers, booleans),
   * serializable maps, or lists thereof.
   *
   * @param form the parameters from the edit HTML form (input)
   * @param errors the collector for error messages (output)
   * @return serializable map of generator-specific parameters
   * @throws GeneratorUnavailableException if the operation cannot be performed
   */
  @NotNull
  Map<String, Object> buildParametersFromForm(@NotNull Map<String, Object> form, @NotNull ErrorCollection errors) throws GeneratorUnavailableException;

  /**
   * <p>Given the generator-specific parameters, fills the parameter map for
   * the summary Velocity template.
   *
   * @param parameters generator parameters (input)
   * @param summaryParameters summary template parameters (output)
   */
  void addParametersForSummary(@NotNull Map<String, Object> parameters, @NotNull Map<String, Object> summaryParameters);

  /**
   * <p>Checks if this generator is currently available in the Automation menu on UI.
   *
   * <p>For example, a generator can be unavailable when a certain plug-in
   * is missing, or a certain application feature is disabled.
   *
   * <p>Note that Structure doesn't stop work of existing generator items if the generator is unavailable.
   * Structure only hides the generator from the Automation menu so users can't create new items from UI.
   *
   * @return true if the generator must be available in the Automation menu
   */
  boolean isAvailable();


  /**
   * An implementor inserts new items into the containing structure.
   */
  @PublicSpi
  interface Inserter extends StructureGenerator { // todo document supplementary interfaces
    /**
     * <p>Builds a forest fragment. The Structure core inserts the fragment right after the row with this Inserter.</p>
     * <h3>Security Note</h3>
     * In spite of {@link com.almworks.jira.structure.api.item.StructureItemType#isVisible(Object, ApplicationUser) StructureItemType#isVisible()}
     * method hides items those a user is not permitted to see, the Inserter implementation ought to check
     * an item visibility before insertion.
     * If the implementation inserts an invisible item, the Structure hides it from the user, but the Structure does not hide
     * the item from a subsequent generator (if any). So the generator may disclose a private data from the item
     * by aggregating it or in some similar way.
     * @param parameters configuration parameters. See {@link #buildParametersFromForm(Map, ErrorCollection)}
     * @param context a generation context
     * @param builder a forest builder
     */
    void createFragment(@NotNull Map<String, Object> parameters, @NotNull GenerationContext context, @NotNull ItemForestBuilder builder);
  }


  @PublicSpi
  interface Extender extends StructureGenerator {
    @Nullable
    ExtenderFunction getExtenderFunction(@NotNull Map<String, Object> parameters, @NotNull GenerationContext context);

    @PublicSpi
    interface ExtenderFunction extends ApplicabilityChecker {
      void extend(@NotNull StructureRow row, @NotNull ItemForestBuilder builder);
    }
  }


  @PublicSpi
  interface Grouper extends StructureGenerator {
    @Nullable
    GrouperFunction createGrouperFunction(@NotNull Map<String, Object> parameters, @NotNull GenerationContext context);

    @Nullable
    String getSemanticLabel(@NotNull Map<String, Object> parameters);

    @PublicSpi
    interface GrouperFunction {
      Set<ItemIdentity> getGroups(@NotNull StructureRow row);
      boolean isExclusiveGroup(@NotNull ItemIdentity group);
      void createGroupRows(Set<ItemIdentity> groups, @NotNull ItemListBuilder builder);
    }
  }


  @PublicSpi
  interface Filter extends StructureGenerator {
    @Nullable
    FilterFunction getFilterFunction(@NotNull Map<String, Object> parameters, @NotNull GenerationContext context);

    @PublicSpi
    interface FilterFunction {
      boolean matches(@NotNull StructureRow row);
    }
  }


  @PublicSpi
  interface Sorter extends StructureGenerator {
    @Nullable
    ApplicabilityChecker getApplicabilityChecker(@NotNull Map<String, Object> parameters, @NotNull Context context);

    @Nullable
    CompareFunction getCompareFunction(@NotNull Map<String, Object> parameters, @NotNull GenerationContext context);

    @PublicSpi
    interface CompareFunction {
      int compare(@NotNull StructureRow row1, @NotNull StructureRow row2);
    }
  }


  @PublicApi
  interface Context {}


  @PublicApi
  interface GenerationContext extends Context {
    /**
     * Detect future changes those may affect the forest with {@link ItemChangeFilter}
     * @param filter the filter to detect changes
     */
    void addItemChangeFilter(ItemChangeFilter filter);

    /**
     * Detect future changes those may affect the forest with {@link UpdateChecker}
     * @param checker the checker to detect changes
     */
    void addUpdateChecker(UpdateChecker checker);

    /**
     * Returns the forest fragment currently processed by the generator.
     */
    @NotNull
    ItemForest previewForest();

    /**
     * Returns the forest fragment generated before current generator call.
     * Null if generator driver does not support this feature.
     */
    @Nullable
    ItemForest inputForest();

    /**
     * @return in case the generation is a transformation, returns forest spec of the forest source being transformed;
     * returns null if the generation is not a transformation (in-structure generation).
     *
     * If multiple transformations take place, the generator running (N+1)th transformation will receive the spec
     * for the result of the Nth transformation.
     */
    @Nullable
    ForestSpec getForestSpecBeingTransformed();

    /**
     * Returns the item version update since the last time the current forest was generated.
     */
    @NotNull
    ItemVersionUpdate itemsUpdate();

    /**
     * Returns the item version update since the given version.
     * @param sinceVersion item data version
     */
    @NotNull
    ItemVersionUpdate itemsUpdate(DataVersion sinceVersion);

    /**
     * Associate an object with the current forest. The object will be kept with the generated forest
     * and available to the current generator during future regenerations, unless the generated forest
     * is evicted from cache or its internal caches are cleared.
     * @param key the key
     * @param value the value
     * @see #getObject(Object)
     */
    void putObject(@Nullable Object key, @Nullable Object value);

    /**
     * Retrieves the object previously associated with the given forest by this generator.
     * May return {@code null} if the generated forest was evicted from cache or its internal
     * caches were cleared.
     * @param key the key
     * @param <T> result type
     * @see #putObject(Object, Object)
     */
    @Nullable
    <T> T getObject(@Nullable Object key);

    /**
     * Associate an object with this context. Unlike {@link #putObject(Object, Object)},
     * the object will be available to this generator only during the current regeneration.
     * @param key the key
     * @param value the value
     * @see #getTempObject(Object)
     * @see #putObject(Object, Object)
     */
    void putTempObject(@Nullable Object key, @Nullable Object value);

    /**
     * Retrieves the object previously associated with this context by this generator.
     * @param key the key
     * @param <T> result type
     * @see #putTempObject(Object, Object)
     */
    @Nullable
    <T> T getTempObject(@Nullable Object key);

    /**
     * Returns the forest spec of the forest being generated.
     */
    @NotNull
    ForestSpec getForestSpec();

    /**
     * Checks if current generation process is stopped
     *
     * @return true iff further generation shouldn't be continued, false otherwise
     */
    boolean isStopped();
  }


  @PublicApi
  interface HandlingContext extends Context {
    void block(String explanation);
    void yield(String explanation);
    void effect(ActionEffect effect, ActionEffect inverse);
    Map<String, Object> parameters();
    void putObject(@Nullable Object key, @Nullable Object value);
    @Nullable <T> T getObject(@Nullable Object key);
  }


  @PublicApi
  interface EffectContext extends Context {
    void block(String explanation);
    void yield(String explanation);
    void effect(String explanation, RunnableE<? extends StructureException> effect);
  }


  @PublicApi
  interface ItemChangeFilterContext extends Context {
    @Nullable <T> T resolveItem(@NotNull ItemIdentity itemId, @NotNull Class<T> itemClass);
  }


  @PublicSpi
  interface ApplicabilityChecker {
    boolean isApplicableTo(StructureRow row);
  }
}
