package com.almworks.jira.structure.api.structure;

import com.almworks.jira.structure.api.auth.StructureAuth;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.ForestService;
import com.almworks.jira.structure.api.permissions.PermissionLevel;
import com.almworks.jira.structure.api.permissions.PermissionRule;
import com.almworks.jira.structure.api.settings.StructureConfiguration;
import com.almworks.jira.structure.api.util.Limits;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.List;

/**
 * <p>This interface lets you inspect and change properties of a structure.</p>
 *
 * <p>Structures are named hierarchical lists of issues and other items; they are the main entities in the Structure add-on.
 * But {@code Structure} interface does not provide access to the hierarchy itself &mdash; that can be done via {@link ForestService}.</p>
 *
 * <p>You typically get an instance of <code>Structure</code> by calling
 * one of the {@link StructureManager} methods. There are <code>get...</code> methods
 * to retrieve the existing structures, and {@link StructureManager#createStructure}
 * method to create a new structure.</p>
 **
 * <p>To change properties of a structure, you need to get the instance of
 * <code>Structure</code>, call some of the <code>set...</code> methods and then
 * call {@link #saveChanges}. Until you call <code>saveChanges()</code> the modifications
 * are made only to the copy of the structure in memory.</p>
 *
 * <p>To create a new structure, call {@link StructureManager#createStructure} to get
 * an empty <code>Structure</code> instance, call <code>set...</code> methods to set
 * properties (at least structure name must be set) and then call {@link #saveChanges}.
 * When <code>saveChanges</code> is completed, the new Structure has been created and
 * you can use {@link #getId()} to get the new structure's ID.</p>
 *
 * <p>Note that the maximum length of a structure name is 190 (see {@link Limits#MAX_NAME_LENGTH}).
 * Longer names will be truncated.</p>
 *
 * <p>Instances of classes implementing this interface have reference equality: their {@code equals()}
 * and {@code hashCode()} behave in the same way as {@code Object}'s.</p>
 *
 * <p>The implementations of this interface are <strong>not thread-safe</strong>. Every thread must obtain their
 * own copy of the <code>Structure</code> instance. Moreover, every time you need an
 * instance of <code>Structure</code>, you should get it from the <code>StructureManager</code>
 * because the properties might have changed or the structure might have been deleted.</p>
 *
 * @author Igor Sereda
 * @see StructureManager
 * @see ForestService
 */
@PublicApi
public interface Structure {
  // ===== GETTERS =====

  /**
   * Returns the ID of the structure, or <code>0</code> if the structure has no ID (when it's
   * new and hasn't been saved yet).
   *
   * @return the ID of the structure, or 0 if it doesn't yet have an ID
   */
  long getId();

  /**
   * Returns the name of the structure or empty string if the name has not been yet set.
   */
  @NotNull
  String getName();

  /**
   * Returns the description of the structure, or empty string if the description has not been set.
   */
  @NotNull
  String getDescription();

  /**
   * Returns the owner of the structure. {@code null} is returned if there's no owner, or if the owner has been
   * removed from JIRA.
   */
  @Nullable
  default ApplicationUser getOwner() {
    return StructureUtil.getUserByKey(getOwnerUserKey());
  }

  /**
   * Returns the user key of the owner of the structure. {@code null} is returned if there's no owner.
   * Note that the user is not guaranteed to exist. Use {@link #getOwner()} to retrieve only an existing user.
   */
  @Nullable
  String getOwnerUserKey();

  /**
   * Returns a list of permissions for this structure. User's access level is defined by applying those rules in the order
   * they are returned from this method. The last matching rule has precedence.
   *
   * @see <a href="http://wiki.almworks.com/display/structure/Structure+Permissions">Structure Permissions (Structure Documentation)</a>
   */
  @NotNull
  List<PermissionRule> getPermissions();


  /**
   * <p>Returns true if modification of the forest requires that the user has <i>Edit Issue</i> permission to the parent
   * issue of the items being moved, added or deleted.</p>
   *
   * <p>Note: currently this flag works only with parent issues. In the future, it may be extended to work with any item
   * types that support edit permission checking.</p>
   *
   * <p>Only the immediate parent issue is checked, higher-level ancestors are not considered.</p>
   *
   * @see <a href="http://wiki.almworks.com/display/structure/Structure+Permissions">Structure Permissions (Structure Documentation)</a>
   */
  boolean isEditRequiresParentIssuePermission();

  /**
   * Returns true if structure is archived.
   */
  boolean isArchived();


  // ===== SETTERS =====


  /**
   * <p>Sets the name of the structure. Although it is possible to set name to null, it is an invalid value
   * and <code>saveChanges</code> will throw an error when called, unless a non-null name is set.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}.</p>
   *
   * <p>Structure name cannot be longer than 190 characters.</p>
   *
   * @param name structure name
   * @return this structure
   */
  @NotNull
  Structure setName(@Nullable String name);


  /**
   * <p>Sets the description of the structure.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}.</p>
   *
   * @param description structure description
   * @return this structure
   */
  @NotNull
  Structure setDescription(@Nullable String description);

  /**
   * <p>Sets the owner of the structure.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}.</p>
   *
   * @param owner new owner
   * @return this structure
   */
  @NotNull
  default Structure setOwner(@Nullable ApplicationUser owner) {
    setOwnerUserKey(StructureUtil.getUserKey(owner));
    return this;
  }

  /**
   * <p>Sets the owner of the structure.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}.</p>
   *
   * @param ownerUserKey new owner's user key
   * @return this structure
   */
  @NotNull
  Structure setOwnerUserKey(@Nullable String ownerUserKey);

  /**
   * <p>Sets the permission rules for this structure. Permission rules are evaluated in the specified order
   * and the last matching rule defines the access level to the structure for the given user.</p>
   *
   * <p>Structure owner and JIRA administrators always have {@link PermissionLevel#ADMIN} access level.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}. At this point, the specified rules are validated, and
   * {@link StructureException} is thrown if any of them fails to validate.</p>
   *
   * @param permissions a collection of permissions, null means empty list
   * @return this structure
   * @see <a href="http://wiki.almworks.com/display/structure/Structure+Permissions">Structure Permissions (Structure Documentation)</a>
   */
  @NotNull
  Structure setPermissions(@Nullable Collection<? extends PermissionRule> permissions);


  /**
   * <p>Changes the security flag that requires the user to have <i>Edit Issue</i> permission on parent
   * issues of the issues being moved, added or deleted from structure.</p>
   *
   * <p>To store the changed information in the database, use {@link #saveChanges}.</p>
   *
   * @param flag if true, <i>Edit Issue</i> permission is required
   * @return this structure
   * @see <a href="http://wiki.almworks.com/display/structure/Structure+Permissions">Structure Permissions (Structure Documentation)</a>
   */
  @NotNull
  Structure setEditRequiresParentIssuePermission(boolean flag);

  /**
   * <p>Changes the archived flag.</p>
   *
   * @param archived if true, the structure is archived
   * @return this structure
   */
  @NotNull
  Structure setArchived(boolean archived);

  /**
   * <p>Call this method to save the changes made with <code>set...</code> methods and update the database.</p>
   *
   * <p>Before this method is called, all changes are only stored in this instance of Structure and have no other
   * effect. After this method has successfully executed, the changes are stored in the database and applied.</p>
   *
   * <p>The changes are done under the current user. Current user can be changed with {@link StructureAuth#sudo}.
   * The updater must have
   * {@link PermissionLevel#ADMIN} access level to the structure, and be allowed to use Structure plugin in
   * {@link StructureConfiguration}. Additionally, if this is a new structure being saved, the user must have
   * permissions to create new structures.</p>
   *
   * <p>All security checks are bypassed if <code>overrideSecurity</code> flag is set via {@code StructureAuth}.</p>
   *
   * <h4>Permission rules validation</h4>
   *
   * <p>Permission rules (set through {@link #setPermissions(java.util.Collection)} or existing if this method was not used) are validated before changing the database.
   * If any of the rules is not valid, StructureException is thrown.</p>

   * @return this structure
   * @throws StructureException if there's a permissions problem, if required fields (name) are not set, or if there
   * are other problems
   */
  @NotNull
  Structure saveChanges() throws StructureException;


  // ===== UTILITIES =====


  /**
   * Calculates permission level for this structure based on the current user.
   * When {@code overrideSecurity} flag is set, this method returns {@code ADMIN}.
   * For instances representing new structures (not yet saved in the database), this method returns {@code NONE}.
   *
   * @return effective permission
   */
  @NotNull
  PermissionLevel getEffectivePermission();
}
