package com.almworks.jira.structure.api.export;

import com.almworks.jira.structure.api.attribute.loader.AttributeContext;
import com.almworks.jira.structure.api.error.StructureProviderException;
import com.almworks.jira.structure.api.view.ViewSpecification;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>{@code ExportRendererProvider} processes the export requests for the Structure widget columns. The required export
 * format is represented by an {@link ExportFormat} instance, and a particular widget column is represented by its
 * {@link ViewSpecification.Column} instance. If a renderer provider recognizes the format and the column, it must
 * return a non-null {@link ExportRenderer} instance that will render the column. The renderer provider may throw a
 * {@link StructureProviderException} when it is supposed to serve the export request, but unable to do so for whatever
 * reason, e.g. missing or invalid column configuration, insufficient permissions, or an unexpected error downstream.
 * </p>
 *
 * <p>In general, all column specifications are given to all interested renderer providers for examination. In
 * {@code atlassian-plugin.xml} the renderer providers can be assigned to specific column keys and/or export formats.
 * The providers with no column key (format) associations are considered generic and used for all column keys (formats).
 * More specific providers have precedence over more generic ones. The providers assigned to the current column key and
 * export format are consulted first, then the providers for the current column key, then the providers for the current
 * export format, then the completely generic providers. This allows the plug-in developers to mix generic and
 * customized export renderers in a flexible way. The order within a specificity class is arbitrary, so if several
 * providers have the same specificity and return {@link ExportRenderer} instances for the same column, it is not
 * defined which one of them will be used.</p>
 *
 * <p>Single-format renderer providers and their renderers can be parameterized with the expected {@link ExportCell} and
 * {@link ExportColumn} subtypes. Generic or multiple-format providers and renderers can use the basic cell and column
 * types or implement the raw interfaces.</p>
 *
 * <p>Each renderer provider is also given a {@link ExportRequestContext} instance that corresponds to the current data
 * request. To render aggregate columns, the provider must call
 * {@link ExportRequestContext#addAttribute(com.almworks.jira.structure.api.aggregate.Aggregate) prepareAggregate()}
 * on the given {@code context}. The result of the aggregate calculation will be available to the returned
 * {@link ExportRenderer} instances via {@link ExportRenderContext#getValues()
 * ExportRenderContext#getPreparedAggregates()}.</p>
 *
 * <p>The provider can also associate arbitrary data with the current request by calling
 * {@link AttributeContext#putObject(Object, Object) putObject()} on the given
 * {@code context}. The data will be available to {@link ExportRenderer} instances via
 * {@link AttributeContext#getObject(Object) ColumnContext.getObject()}. Please
 * note that contexts are shared between {@code ExportRendererProvider} and {@code ExportRenderer} instances from all
 * plug-ins, so we advise that you use unique keys (e.g. private singleton objects or your own {@code enum}s) to avoid
 * conflicts.</p>
 *
 * todo update docs

 * @param <C> The expected cell type.
 * @param <M> The expected column type.
 *
 * @see ExportRenderer
 */
@PublicSpi
public interface ExportRendererProvider<C extends ExportCell, M extends ExportColumn<C>> {
  /**
   * @param exportFormat The export format.
   * @param columnSpec The column specification.
   * @param context The current request context.
   * @return An {@link ExportRenderer} instance if this renderer provider can serve the given {@code columnSpec},
   * {@code null} otherwise.
   * @throws StructureProviderException if this provider is supposed to serve the given column, but cannot serve it, e.g.
   * because of missing or invalid parameters, insufficient permissions, or for any other reason.
   */
  @Nullable
  ExportRenderer<C, M> getColumnRenderer(
    @NotNull ExportFormat exportFormat,
    @NotNull ViewSpecification.Column columnSpec,
    @NotNull ExportRequestContext context)
    throws StructureProviderException;
}
