package com.almworks.jira.structure.api.forest;

import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.action.*;
import com.almworks.jira.structure.api.forest.raw.Forest;
import com.almworks.jira.structure.api.pull.DataVersion;
import com.almworks.jira.structure.api.pull.VersionedDataSource;
import org.jetbrains.annotations.NotNull;

import java.util.Collections;
import java.util.Map;

/**
 * <p>{@code ForestSource} is used to retrieve a forest and get forest updates.</p>
 *
 * <p>A forest is a core concept in Structure &mdash; structures and queries all generate a forest, which is then
 * shown in the forest panels in the Structure Widget.</p>
 *
 * <p>A {@code ForestSource} is a pull-based source of data, using versioning to deliver full or incremental
 * updates to the caller. See {@link VersionedDataSource} for the general concept of versioned data source.</p>
 *
 * <p>You can get an instance of {@code ForestSource} from {@link ForestService} by providing a {@link ForestSpec}.
 * The instances of {@code ForestSource} can be cached by Structure and shared among different threads and callers.
 * However, the caller should not store {@code ForestSource} for an extended period of time &mdash; normally,
 * an instance must be requested from {@code ForestService} every time a new request comes.</p>
 *
 * <h3>State</h3>
 *
 * <p>The full state of the {@code ForestSource} is represented by {@link VersionedForest}, which is just a
 * {@link Forest} tagged with a {@link DataVersion}. When applying full update, the caller should replace the
 * old forest with the new {@code VersionedForest}.</p>
 *
 * <p>The incremental update is represented as a list of {@link ForestChange} objects, each describing an atomic
 * change. When applying incremental update, forest changes must be applied to the old state one by one in the
 * correct order.</p>
 *
 * <h3>Actions</h3>
 *
 * <p>Most {@code ForestSource} instances support actions &mdash; changes that are applied to the source of the
 * forest such as items addition or movement. {@link ForestAction} abstract class and its sub-classes define
 * the range of possible actions. They are really basic: add, move, remove or copy specific rows in the forest.</p>
 *
 * <p>To apply the action, call {@link #apply(ForestAction)} method. A successful application of an action may
 * lead to various effects &mdash; represented by {@link AppliedEffect}. In a fully static structure, without
 * dynamic content, all actions will update the contents of the structure. In a dynamic structure, a forest
 * action applied to the dynamic content may actually change something like JIRA issue values, links or anything
 * else, in such a way that the next forest update will show the desired result (item moved, added or removed).</p>
 *
 * @see ForestService
 * @see VersionedDataSource
 * @see VersionedForestUpdate
 */
public interface ForestSource extends VersionedDataSource<VersionedForestUpdate> {
  /**
   * Returns a snapshot of the current state.
   */
  @NotNull
  VersionedForest getLatest();

  /**
   * <p>This method allows to quickly check if the caller has the last version of the forest.</p>
   *
   * <p>Although the same can be done by just requesting an update, this method can be optimized by the
   * Structure engine.</p>
   *
   * <p>Note that if this method returns {@code true}, it is not guaranteed that there will be a non-empty update.</p>
   *
   * @param sinceVersion current version of the forest that the caller has
   * @return {@code false} if the version is the most recent, i.e. calling {@link #getUpdate(DataVersion)} would
   * definitely result in an empty update; {@code true} if the version specified is <em>probably</em> not the most
   * recent, i.e. calling {@link #getUpdate(DataVersion)} <em>maybe</em> would result in an update.
   */
  boolean hasUpdate(@NotNull DataVersion sinceVersion);

  /**
   * <p>Applies a forest action to this source. If the method returns and does not throw an exception, the action
   * has been successful and the outcome can be retrieved from the returned {@link ActionResult}.</p>
   *
   * <p>The method throws {@link StructureException} if the action cannot be applied. The exception object can be
   * inspected for additional information about the problem &mdash; use {@link StructureException#getLocalizedMessage()}
   * for a (hopefully) user-friendly message, and {@link StructureException#getRow()} to identify the row ID that is
   * associated with the problem.</p>
   *
   * <p>A special case of the action application problem is represented by {@link StructureInteractionException}.
   * This exception, a sub-class of {@code StructureException}, is thrown when Structure needs additional information
   * from the user to execute an action. It can be a decision about which generator must process the action, or a
   * confirmation from the user about a bulk action, or something else. The calling code may figure out the answers
   * and apply the action again, passing the answers in {@code parameters}.</p>
   *
   * @param action the action to apply
   * @param parameters additional parameters. Some parameters are defined in {@link ActionParameters}.
   * @return the result of successful application of the action
   * @throws StructureInteractionException if additional information is needed from the actor
   * @throws StructureException if the action cannot be executed
   */
  @NotNull
  ActionResult apply(ForestAction action, Map<String, Object> parameters) throws StructureException;

  /**
   * Convenient method to call {@link #apply(ForestAction, Map)} with an empty parameter map.
   *
   * @param action the action to apply
   * @return the result of successful application of the action
   * @throws StructureInteractionException if additional information is needed from the actor
   * @throws StructureException if the action cannot be executed
   */
  default ActionResult apply(ForestAction action) throws StructureException {
    return apply(action, Collections.emptyMap());
  }

  /**
   * Checks if the forest source supports actions in principle. A {@code false} result means that any call to {@link #apply}
   * would fail. A {@code true} result means that some actions may succeed.
   *
   * @return {@code false} if it is guaranteed that {@code apply()} methods would fail every time
   */
  default boolean isActionable() { return true; }

  /*
   * Thoughts for the future:
   * Consider the fact that forest source may become invalid (and valid again).
   * For example, the user loses access to a structure, or JQL becomes invalid.
   * This suggests that ForestService may have the contract that *always* provides a ForestSource for
   * a given spec, but a ForestSource may become invalid or valid again.
   *
   * Try this idea in Structure 4 or later:
   *     ForestSourceHealthStatus getHealth();
   *
   * todo: Can return outdated or incorrect results, please use getUpdate().getHealth() instead.
   */
  default ForestSourceHealthStatus getHealth() {
    return ForestSourceHealthStatus.HEALTHY;
  }
}
