package com.almworks.jira.structure.api.item.generic;

import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.bc.ServiceResultImpl;
import com.atlassian.jira.util.ErrorCollection;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p>{@code GenericItemService} is responsible for performing get, create, update and delete operations
 * with generic items.</p>
 *
 * <p>Corresponding validation should be performed before all operations except get.</p>
 *
 * <p>Along with data related to operation each result may contain an error collection with any error messages
 * that may have been generated when performing the operation.
 * Also result may contain exception that has been thrown during operation execution.</p>
 *
 * @see GenericItemManager
 * @see GenericItem
 */
@PublicApi
public interface GenericItemService {
  @NotNull
  GenericItemResult getItem(@NotNull ItemIdentity itemId);

  @NotNull
  CreateValidationResult validateCreate(@NotNull String itemType, @NotNull GenericItem genericItem);

  @NotNull
  GenericItemResult create(@NotNull CreateValidationResult validationResult);

  @NotNull
  UpdateValidationResult validateUpdate(@NotNull ItemIdentity itemId, @NotNull GenericItem genericItem);

  @NotNull
  GenericItemResult update(@NotNull UpdateValidationResult validationResult);

  @NotNull
  DeleteValidationResult validateDelete(@NotNull ItemIdentity itemId);

  @NotNull
  GenericItemResult delete(@NotNull DeleteValidationResult validationResult);

  abstract class ExceptionSupportResult extends ServiceResultImpl {
    private final StructureException myException;

    ExceptionSupportResult(@NotNull ErrorCollection errorCollection, @Nullable StructureException exception) {
      super(errorCollection);
      myException = exception;
    }

    @Override
    public boolean isValid() {
      return super.isValid() && myException == null;
    }

    @Nullable
    public StructureException getException() {
      return myException;
    }
  }

  class GenericItemResult extends ExceptionSupportResult {
    private final ItemIdentity myItemId;
    private final GenericItem myGenericItem;

    public GenericItemResult(@NotNull ErrorCollection errors, @Nullable StructureException exception,
      @Nullable ItemIdentity itemId, @Nullable GenericItem genericItem)
    {
      super(errors, exception);
      myItemId = itemId;
      myGenericItem = genericItem;
    }

    @Nullable
    public ItemIdentity getItemId() {
      return myItemId;
    }

    @Nullable
    public GenericItem getGenericItem() {
      return myGenericItem;
    }
  }

  class CreateValidationResult extends ExceptionSupportResult {
    private final String myItemType;
    private final GenericItem myGenericItem;

    public CreateValidationResult(@NotNull ErrorCollection errors, @Nullable StructureException exception,
      @NotNull String itemType, @NotNull GenericItem genericItem)
    {
      super(errors, exception);
      myItemType = itemType;
      myGenericItem = genericItem;
    }

    @NotNull
    public String getItemType() {
      return myItemType;
    }

    @NotNull
    public GenericItem getGenericItem() {
      return myGenericItem;
    }
  }

  class UpdateValidationResult extends ExceptionSupportResult {
    private final ItemIdentity myItemId;
    private final GenericItem myGenericItem;

    public UpdateValidationResult(@NotNull ErrorCollection errors, @Nullable StructureException exception,
      @NotNull ItemIdentity itemId, @NotNull GenericItem genericItem)
    {
      super(errors, exception);
      myItemId = itemId;
      myGenericItem = genericItem;
    }

    @NotNull
    public ItemIdentity getItemId() {
      return myItemId;
    }

    public GenericItem getGenericItem() {
      return myGenericItem;
    }
  }

  class DeleteValidationResult extends ExceptionSupportResult {
    private final ItemIdentity myItemId;

    public DeleteValidationResult(@NotNull ErrorCollection errors, @Nullable StructureException exception,
      @NotNull ItemIdentity itemId)
    {
      super(errors, exception);
      myItemId = itemId;
    }

    @NotNull
    public ItemIdentity getItemId() {
      return myItemId;
    }
  }
}