package com.almworks.jira.structure.api.pull;

import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

/**
 * Base class for versioned updates.
 *
 * @see VersionedDataSource
 */
@PublicApi
public abstract class VersionedDataUpdate {
  private final DataVersion myVersion;

  protected VersionedDataUpdate(@NotNull DataVersion version) {
    //noinspection ConstantConditions
    if (version == null) throw new IllegalArgumentException("version must not be null");
    myVersion = version;
  }

  /**
   * Returns the version associated with this update.
   */
  @NotNull
  public final DataVersion getVersion() {
    return myVersion;
  }

  /**
   * Returns true if the update is "full", that is, the caller may drop the current state and replace it fully
   * with the state from this update.
   */
  public boolean isFull() {
    return false;
  }

  /**
   * Returns true if the update is "empty", that is, the caller may keep the state, because it is up-to-date.
   */
  public boolean isEmpty() {
    return false;
  }

  /**
   * Returns true if the update is "incremental", that is, the caller may get the up-to-date state by applying
   * a diff, contained in this update, to the previous state they have.
   */
  public boolean isIncremental() {
    return !isFull();
  }
}
