package com.almworks.jira.structure.api.statistics;

import com.atlassian.annotations.Internal;
import org.jetbrains.annotations.NotNull;

@Internal
public interface StructureStatisticsRecorder {
  /**
   * Counts that the current logged-in user has used the specified feature.
   * The resulting statistic will be the average amount of unique feature users per day.
   * <p>
   * You may also want to count the average total amount of feature usages per day via {@link #addTotalCountAsync(String)}.
   * Don't use the same feature ID for both total and unique user counts.
   * <p>
   * The "async" part in the method name means that this object aggregates statistics and writes them to the backend in bulk.
   * */
  void addUniqueUserCountAsync(@NotNull String feature);

  /**
   * Counts that the specified feature was used.
   * The resulting statistic will be the average amount of times the feature is used per day.
   * <p>
   * You may also want to count the average total amount of feature usages per day via {@link #addUniqueUserCountAsync(String)}.
   * Don't use the same feature ID for both total and unique user counts.
   * <p>
   * The "async" part in the method name means that this object aggregates statistics and writes them to the backend in bulk.
   * */
  void addTotalCountAsync(@NotNull String feature);

  /**
   * Adds the specified value to the specified statistic.
   * The resulting statistic will be the average sum of recorded values per day.
   * This sum, or "count", is "total" in the sense that it does not account for the distribution of these values on users.
   * <p>
   * The "async" part in the method name means that this object aggregates statistics and writes them to the backend in bulk.
   * */
  void addTotalCountAsync(@NotNull String stat, double value);

  /**
   * Adds 1 to the statistics with the ID equal to {@code statPrefix.<bin>}, where {@code bin} is the least element
   * of the {@code bins} array greater than {@code value} or {@code "inf"} if all of the elements are less than {@code value}.
   * <p>
   * In other words, if there are {@code n} bins {@code b_0, ... b_{n-1}}, this method adds 1 to one of the {@code n + 1}
   * statistics, which correspond to these intervals:
   * <table>
   *   <tr><th>statistic</th> <td>statPrefix.b_0</td> <td>statPrefix.b_1</td> <td>...</td> <td>statPrefix.b_{n-1}</td> <td>statPrefix.inf</td></tr>
   *   <tr><th>interval</th>  <td>(-\inf, b_0]</td>   <td>(b_0 b_1]</td>      <td>...</td> <td>(b_{n-2} b_{n-1}]</td> <td>(b_{n-1}, \inf)</td></tr>
   * </table>
   * <p>
   * The resulting statistics sharing {@code statPrefix} contain the distribution of the measured integer value over
   * the specified bins, averaged per day.
   * <p>
   * The "async" part in the method name means that this object aggregates statistics and writes them to the backend in bulk.
   * */
  void addToIntValueDistrAsync(@NotNull String statPrefix, int value, @NotNull int... bins);

  /**
   * Adds statistic source which is queried periodically for the statistic data
   * @param source source to register
   */
  void addStatisticSource(StatisticSource source);

  /**
   * Removes statistic source which was previously registered
   * @param source source to unregister
   */
  void removeStatisticSource(StatisticSource source);

  /**
   * Adds 1 to use count for {@code feature} (like {@link #addTotalCountAsync(String)}) and
   * adds the current user to {@code activeUser.<feature>} (like {@link #addUniqueUserCountAsync(String)}).
   */
  void addTotalCountAndActiveUserAsync(@NotNull String feature);
}
