package com.almworks.jira.structure.api.sync;

import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.atomic.AtomicReference;

/**
 * <p><code>SyncInstance</code> represents an instance of a synchronizer, configured with user-specified parameters
 * to act on some structure under some user account. It may be either installed on that structure or  
 * represent a {@link StructureSyncManager#resyncOnce one-time synchronization action}. It can represent an 
 * instance of a synchronizer that is not currently available, e.g., when the module which defines it is not loaded;
 * then and only then {@link #getSynchronizer()} returns {@code null}.</p>
 *
 * <p>The synchronizer receives <code>SyncInstance</code> parameter whenever it is asked
 * to do an instance-specific task such as syncing.</p>
 *
 * @author Igor Sereda
 */
@PublicApi
public interface SyncInstance {
  /**
   * Returns the ID of this synchronizer instance. It is 0 iff this sync instance represents a one-time resync.
   */
  long getId();

  /**
   * Returns the synchronizer for this sync instance. {@code null} means that the synchronizer is currently
   * unavailable (the module that defines it is not loaded).
   */
  @Nullable
  StructureSynchronizer getSynchronizer();
  
  /**
   * Returns module key of the synchronizer for this sync instance. Always non-{@code null}, even if the synchronizer
   * is currently unavailable.
   * */
  @NotNull
  String getSynchronizerModuleKey();  

  /**
   * Returns the ID of the structure this sync instance is acting on.
   */
  long getStructureId();

  /**
   * Synchronizer parameters. The type of the object is controlled by the
   * synchronizer's {@link StructureSynchronizer#buildParametersFromForm}
   * and {@link StructureSynchronizer#restoreParameters} methods.
   *
   * @return the parameters of the synchronization
   */
  @Nullable
  Object getParameters();

  /**
   * Returns user key of the user account that is used to run synchronizer jobs with this sync instance.
   */
  @NotNull
  String getUserKey();

  /**
   * <p>Provides the synchronizer with a temporary / cache storage.</p>
   * <p>
   * The <code>key</code> parameter is an arbitrary string that the synchronizer can use 
   * to have storage for several types of the information. The values are stored for this sync instance only.</p>
   * <p>
   * Note: in JIRA Data Center, the cache is local to the current node; 
   * changes to the cached data on one node are not visible by other nodes.</p>
   * <p>
   * Also note that the cache lifetime is shorter than the lifetime of this {@code SyncInstance}: it may be 
   * cleared, for example, when someone modifies the parameters of this instance. 
   * Cache may also be cleared in other circumstances, like when there are many synchronizer instances in the system.</p>
   *
   * @param key data key
   * @param <T> type of data stored
   * @return the container for the stored data
   */
  @NotNull
  <T> AtomicReference<T> transientData(String key);
}

