package com.almworks.jira.structure.api.forest.action;

import com.almworks.integers.LongList;
import com.almworks.integers.LongLongMap;
import com.almworks.integers.wrappers.LongLongHppcOpenHashMap;
import com.almworks.jira.structure.api.error.StructureException;
import com.almworks.jira.structure.api.forest.item.*;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.Immutable;

@PublicApi
public abstract class ForestAction {
  private ForestAction() {}

  public abstract void accept(Visitor visitor) throws StructureException;

  @PublicApi
  @Immutable
  public static final class Remove extends ForestAction {
    private final LongList myRowIds;

    public Remove(LongList rowIds) {
      myRowIds = rowIds;
    }

    public LongList getRowIds() {
      return myRowIds;
    }

    @Override
    public void accept(Visitor visitor) throws StructureException {
      visitor.visit(this);
    }

    @Override
    public String toString() {
      return "remove:" + myRowIds;
    }
  }

  // todo document behaviour in case of not normalized input (e.g. a parent and a child)
  @PublicApi
  @Immutable
  public static final class Move extends ForestAction {
    private final LongList myRowIds;
    private final long myUnder;
    private final long myAfter;
    private final long myBefore;

    public Move(LongList rowIds, long under, long after, long before) {
      myRowIds = rowIds;
      myUnder = under;
      myAfter = after;
      myBefore = before;
    }

    public LongList getRowIds() {
      return myRowIds;
    }

    public long getUnder() {
      return myUnder;
    }

    public long getAfter() {
      return myAfter;
    }

    public long getBefore() {
      return myBefore;
    }

    @Override
    public void accept(Visitor visitor) throws StructureException {
      visitor.visit(this);
    }

    @Override
    public String toString() {
      return "move:" + myRowIds + ":" + myUnder + ":" + myAfter + ":" + myBefore;
    }
  }


  @PublicApi
  @Immutable
  public static final class Add extends ForestAction {
    private final ItemForest myFragment;
    private final long myUnder;
    private final long myAfter;
    private final long myBefore;

    public Add(@NotNull ItemForest fragment, long under, long after, long before) {
      myFragment = ImmutableItemForest.copy(fragment);
      myUnder = under;
      myAfter = after;
      myBefore = before;
    }

    public Add(@NotNull ItemIdentity itemId, long semantics, long under, long after, long before) {
      this(new ItemForestBuilderImpl().nextRow(itemId, semantics).build(), under, after, before);
    }

    public Add(@NotNull ItemIdentity itemId, long under, long after, long before) {
      this(itemId, 0, under, after, before);
    }

    @NotNull
    public ItemForest getFragment() {
      return myFragment;
    }

    public long getUnder() {
      return myUnder;
    }

    public long getAfter() {
      return myAfter;
    }

    public long getBefore() {
      return myBefore;
    }

    @Override
    public void accept(Visitor visitor) throws StructureException {
      visitor.visit(this);
    }

    @Override
    public String toString() {
      return "add:" + myFragment.getForest() + ":" + myUnder + ":" + myAfter + ":" + myBefore;
    }
  }


  @PublicApi
  @Immutable
  public static final class Copy extends ForestAction {
    private final ItemForest myFragment;
    private final long myUnder;
    private final long myAfter;
    private final long myBefore;
    private final LongLongMap myOriginalRows;

    public Copy(ItemForest fragment, long under, long after, long before, LongLongMap originalRows) {
      myFragment = ImmutableItemForest.copy(fragment);
      myUnder = under;
      myAfter = after;
      myBefore = before;
      LongLongHppcOpenHashMap map = new LongLongHppcOpenHashMap(originalRows.size());
      map.putAll(originalRows);
      myOriginalRows = map;
    }

    public ItemForest getFragment() {
      return myFragment;
    }

    public long getUnder() {
      return myUnder;
    }

    public long getAfter() {
      return myAfter;
    }

    public long getBefore() {
      return myBefore;
    }

    public LongLongMap getOriginalRows() {
      return myOriginalRows;
    }

    @Override
    public void accept(Visitor visitor) throws StructureException {
      visitor.visit(this);
    }

    @Override
    public String toString() {
      return "copy:" + myFragment.getForest() + ":" + myUnder + ":" + myAfter + ":" + myBefore + ":" + myOriginalRows;
    }
  }


  @PublicSpi
  public interface Visitor {
    void visit(@NotNull Remove remove) throws StructureException;
    void visit(@NotNull Move move) throws StructureException;
    void visit(@NotNull Add add) throws StructureException;
    void visit(@NotNull Copy copy) throws StructureException;
  }
}
