package com.almworks.jira.structure.api.forest.raw;

import com.almworks.integers.*;
import com.almworks.jira.structure.api.query.StructureQuery;
import com.almworks.jira.structure.api.row.*;
import com.almworks.jira.structure.api.util.La;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

/**
 * <p><code>Forest</code> interface provides indexed access to a data structure that stores row hierarchy
 * (the hierarchy is a sequence of tree structures, a forest).
 * That is, you can read row IDs and depths given that you know the index of the pair in the
 * sequence. There are also a number of utility methods that allow searching and browsing the
 * forest.
 * </p>
 *
 * <p>To get the details of a specific row (such as issue ID), you need to obtain
 * {@link StructureRow} from {@link RowManager}.</p>
 *
 * <p>
 * Forest of rows is represented as a sequence of pairs <code>(long row, int depth)</code>. The hierarchy
 * is identified by the position in the sequence and the associated depth. For example, the following hierarchy:
 * <pre>
 *     A
 *        A1
 *        A2
 *     B
 *        B1
 *            B1X
 *   </pre>
 * is represented in the forest as the following sequence:
 * <pre>
 *     (A, 0), (A1, 1), (A2, 1), (B, 0), (B1, 1), (B1X, 2)
 *   </pre>
 * </p>
 *
 * <h3>Invariants</h3>
 * <p>
 * Because <code>depth</code> associated with a row is an integer number, there are certain
 * invariants about the depth that must hold true, otherwise the data does not make sense. Additionally,
 * a row may appear no more than once in the forest. Make sure
 * you don't violate those invariants when creating new forest instances:
 * </p>
 * <ul>
 * <li>The depth of the first element in the forest must be <code>0</code>.</li>
 * <li>Given that the depth of an element is <code>D</code>,
 * the depth of the following element must be within range <code>[0 .. D+1]</code>.
 * </li>
 * <li>If a row is present in the forest at index <code>N</code>, it must not be present at any other index.
 * </li>
 * <li>Row ID may not be <code>0</code> - this value represents "missing" or "no value" in various methods of this interface.</li>
 * </ul>
 * <p>
 * When Java assertions are enabled, these invariants are checked all the time. Due to performance impact,
 * the invariants are not checked and assumed to hold true when running with assertions disabled.
 * The result of creating and using a forest that violates these invariants is undefined.
 * </p>
 *
 * <h3>Super-root</h3>
 *
 * <p>The super-root row ({@link SuperRootRow}) has a predefined row ID of {@code -1}. This row id is reserved
 * and cannot be used for normal rows.</p>
 *
 * @author Igor Sereda
 */
public interface Forest extends LongIntIterable {
  Forest EMPTY = new ArrayForest().makeImmutable();

  // ======================== ACCESSORS - WORK FAST ========================
  /**
   * Returns the size of the forest, the number of rows in it, the number is always >= 0.
   * 
   * @return the size of the forest
   */
  int size();

  /**
   * Used to check if the forest does not contain any rows.
   * 
   * @return true if the size of the forest is zero
   */
  boolean isEmpty();

  /**
   * Returns a non-modifiable list of rows, in the order they appear in the forest. The size of
   * the list is equal to the value returned by {@link #size()}.
   * 
   * @return list of IDs of rows contained in the forest
   */
  @NotNull
  LongList getRows();

  /**
   * Returns a non-modifiable list of depths, in the order the rows appear in the forest. The size
   * of the list is equal to the value returned by {@link #size()} and the <code>i</code>-th element
   * of this list corresponds to the row ID at the <code>i</code>-th position in the list returned
   * by {@link #getRows()}.
   * 
   * @return list of depths of rows contained in the forest, parallel to the rows list
   */
  @NotNull
  IntList getDepths();

  /**
   * Gets the ID of the row at the specified position in the forest.
   *
   * @param index the index of the forest entry
   * @return row ID at the specified index
   * @throws IndexOutOfBoundsException if <code>index</code> is not within range <code>[0 .. size() - 1]</code>.
   */
  long getRow(int index);

  /**
   * Gets the depth of the row at the specified position in the forest.
   *
   * @param index the index of the forest entry
   * @return depth at the specified index, value is always >= 0 and satisfies the invariants for the {@link Forest}
   * @throws IndexOutOfBoundsException if <code>index</code> is not within range <code>[0 .. size() - 1]</code>.
   */
  int getDepth(int index);

  /**
   * Returns true if the forest is immutable.
   *
   * @see ArrayForest#makeImmutable()
   */
  boolean isImmutable();


  // ======================== SEARCHERS - WORK SLOWER ========================

  /**
   * <p>Searches for the position of a specific row in the forest.</p>
   *
   * <p>Execution of this method may take <code>O(size())</code> time, however it may be
   * optimized if the implementation maintains an index. It's better to use this method rather
   * than using <code>forest.getRows().indexOf(row)</code> because of the possible optimizations.</p>
   *
   * @param row the row ID to search for
   * @return such index of the row in the forest, or <code>-1</code> if the row is not found
   */
  int indexOf(long row);

  /**
   * <p>Can be used to check if the forest contains a specific row.</p>
   *
   * <p>Execution of this method may take <code>O(size())</code> time.</p>
   *
   * @param row the row ID to search for
   * @return true if the forest contains the specified row
   */
  boolean containsRow(long row);

  /**
   * <p>The method looks for the end of a subtree rooted at the specified index.</p>
   *
   * <p>A subtree rooted at index <code>k</code> is a sub-sequence in the forest starting at position <code>k</code>
   * and containing all following elements that have depth <code>d > depth[k]</code>.</p>
   *
   * <p>The result of this method is the <strong>next</strong> index after the last element of this subsequence.
   * More specifically, the result is the first element after <code>[k]</code> that has depth <code>d <= depth[k]</code>.</p>
   *
   * <p>When the subtree ends with the whole forest, the return value is equal to {@link #size()}.</p>
   *
   * <p>If <code>index</code> is a negative number, the returned value is <code>0</code>. If <code>index</code> is greater or equal than
   * <code>size()</code>, the returned value is equal to <code>size()</code>.</p>
   *
   * @param index the index of the root row of the subtree
   * @return the index of the row that follows the last row of the subtree, or the size of the forest in case the subtree is at the end of it
   */
  int getSubtreeEnd(int index);

  /**
   * <p>Searches the forest for the index of a "parent row". If the row at the specified index is a root row (has depth of <code>0</code>),
   * returns -1.</p>
   *
   * <p>If <code>index</code> is negative, returns -1.</p>
   *
   * @param index index of a child row
   * @return -1 if the row at <code>index</code> is a root row, or index <code>k</code> of the parent row, such as
   * that <code>k = MAX(k in [0, index-1] that has depth[k] == depth[index] - 1)</code>
   * @throws IndexOutOfBoundsException if <code>index</code> is equal or greater than forest size
   */
  int getParentIndex(int index);

  /**
   * Gets the parent row of the specified row.
   *
   * @param row child row
   * @return parent row, or <code>0</code> if the child row is a root row or not found in the forest
   */
  // todo should return -1 or something other than 0 if the row is not in the forest - can be mixed with the result from top-level
  long getParent(long row);

  /**
   * Gets the row that immediately precedes the specified row in the
   * list of children of the specified row's parent.
   *
   * @param row a row
   * @return immediately preceding sibling (row that has the same parent and same depth), 
   * or 0 if <code>row</code> is not found in the forest or is the first root or the first child of its parent row
   */
  long getPrecedingSibling(long row);

  /**
   * Gets the index of the row that immediately precedes the row at the given index in the
   * list of children of its parent.
   *
   * @param index row index
   * @return index of immediately preceding sibling, or -1 if row at <code>index</code>
   * is not found in the forest or is the first root or the first child of its parent row
   */
  int getPrecedingSiblingIndex(int index);

  /**
   * Gets the row that immediately precedes the one with the given index in the
   * list of children of its parent.
   *
   * @param index row index
   * @return immediately preceding sibling, or 0 if there is none
   */
  long getPrecedingSiblingForIndex(int index);

  /**
   * Returns the array of all rows that come before the given row in the list of children of its parent,
   * in the same order as they appear in the forest.
   *
   * @param row a row
   * @return array of all preceding siblings, empty if <code>row</code> has none or is not in the forest
   */
  @NotNull
  LongArray getPrecedingSiblings(long row);

  /**
   * Returns the array of all rows that come before the given row in the list of children of its parent,
   * in the same order as they appear in the forest.
   *
   * @param index row index
   * @return array of all preceding siblings, empty if row at <code>index</code> has none or <code>index</code> is negative
   */
  @NotNull
  LongArray getPrecedingSiblingsForIndex(int index);

  /**
   * Gets the index of the row that immediately follows the row at the given index in the
   * list of children of its parent.
   *
   * @param index row index
   * @return index of immediately following sibling, or -1 if row at <code>index</code>
   * is not found in the forest or is the last root or the last child of its parent row
   */
  int getNextSiblingIndex(int index);

  /**
   * Gets the row that immediately follows the one with the given index in the
   * list of children of its parent.
   *
   * @param index row index
   * @return immediately following sibling, or 0 if there is none
   */
  long getNextSiblingForIndex(int index);

  /**
   * Gets the row that immediately follows the specified row in the
   * list of children of the specified row's parent.
   *
   * @param row a row
   * @return immediately following sibling (row that has the same parent and same depth),
   * or 0 if <code>row</code> is not found in the forest or is the last root or the last child of its parent row
   */
  long getNextSibling(long row);

  /**
   * <p>Given row at the specified index, traverses its "path" upwards - that is, looks for all
   * parent rows up to the topmost root parent, and returns an index of the parent that has the
   * specified depth.</p>
   *
   * <p>If the required depth is equal to the depth of the row at the specified index, returns <code>index</code>.</p>
   *
   * <p>If <code>index</code> is a negative value, returns -1. If row at the specified index has less depth than the required value, returns <code>-1</code>.</p>
   *
   * @param index index of the row
   * @param depth required depth of the [grand-] parent row
   * @return the index of the [grand-] parent row on the "path" to the specified row that has the specified depth, or <code>-1</code> if not found
   * @throws IndexOutOfBoundsException if index is equal or greater than the size of the forest
   */
  int getPathIndexAtDepth(int index, int depth);

  /**
   * <p>Gets the last direct sub-row of the specified parent row.</p>
   *
   * <p><strong>Special case</strong>: when <code>parent</code> is <code>0</code>,
   * returns the last root row in the forest.</p>
   *
   * <p>If the parent row is not in the forest, or if it does not have child rows, the return value is <code>0</code>.</p>
   *
   * @param parent parent row
   * @return the last row among the parent row's children, or 0 if the parent row does not have children
   */
  long getLastChild(long parent);

  /**
   * <p>Gets the last direct sub-row of the specified parent row.</p>
   *
   * <p><strong>Special case</strong>: when <code>parentIndex</code> is less than zero,
   * returns the last root row in the forest.</p>
   *
   * <p>If the parent row does not have child rows, the return value is <code>0</code>.</p>
   *
   * @param parentIndex the index of the parent row
   * @return the last row among the parent row's children, or 0 if the parent row does not have children
   */
  long getLastChildByIndex(int parentIndex);

  /**
   * <p>Returns an iterator over indices of all direct sub-rows of the row at the specified index.</p>
   *
   * <p>This method is a lazy variant of {@link #getChildrenAtIndex(int)}: the underlying data structure is scanned
   * as you advance the returned iterator.  This allows to save on an extra scan, and on copying to the result array.</p>
   *
   * <p>Another difference is that this method returns indices, not the rows themselves.</p>
   *
   * @param index the index of the parent row, in the interval [-1; forest.size()); if -1, will iterate over roots
   * @return iterator over indices of all sub-rows of the specified row that have depth equal to the parent depth + 1, or
   *   over indices of roots if index is -1
   * @throws IndexOutOfBoundsException if the index is greater or equals the size of the forest
   * */
  @NotNull
  IntIterator getChildrenIndicesIterator(int index);

  // ======================== COPIERS - create new data structures with data from the forest ========================

  /**
   * <p>Creates an array with all direct sub-rows of the specified row.</p>
   *
   * <p>The returned array is writable and owned by the calling code.</p>
   *
   * <p>If row is not in the forest or does not have children, empty array is returned.</p>
   *
   * @param row the parent row, you can pass 0 to get the roots.
   * @return array of all sub-rows of the specified row that have depth equal to the parent depth + 1
   */
  @NotNull
  LongArray getChildren(long row);

  /**
   * <p>Creates an array with all direct sub-rows of the row at the specified index.</p>
   *
   * <p>The returned array is writable and owned by the calling code.</p>
   *
   * <p>If the specified row does not have children, or if the index is negative, empty array is returned.</p>
   *
   * @param index the index of the parent row
   * @return array of all sub-rows of the specified row that have depth equal to the parent depth + 1
   * @throws IndexOutOfBoundsException if the index is greater or equals the size of the forest
   */
  @NotNull
  LongArray getChildrenAtIndex(int index);

  /**
   * Returns an array of all root rows in the forest (those that have depth of <code>0</code>).
   * 
   * @return an array of all root rows in the forest (those that have depth of <code>0</code>)
   */
  @NotNull
  LongArray getRoots();

  /**
   * <p>Returns the path to the specified row - a sequence of rows that starts with a
   * root row, ends with the specified row, and where <code>element[i]</code> is
   * the parent row of <code>element[i+1]</code>.</p>
   *
   * <p>If <code>row</code> is not in the forest, returns empty array.</p>
   *
   * <p>The array is modifiable and owned by the calling code.</p>
   *
   * @param row row to get the path to
   * @return path to the specified row, or empty array if the row is not in the forest
   */
  @NotNull
  LongArray getPath(long row);

  /**
   * Similar to {@link #getPath}, returns the path to the row specified by index.
   *
   * @param index the index of the row to get the path to
   * @return path to the specified row, or empty array if the index is negative
   * @throws java.util.NoSuchElementException if the index is greater than or equal to forest size
   */
  @NotNull
  LongArray getPathForIndex(int index);

  /**
   * <p>Returns the path to the specified row without the row itself. In other words, this
   * is the path to the parent of the specified row, if there is one.</p>
   *
   * @param index row index
   * @return path to the specified row's parent; empty array if the row is top-level or not in the forest
   * @see #getPath(long)
   */
  @NotNull
  LongArray getParentPathForIndex(int index);

  /**
   * <p>Filters this forest by hiding rows that do not pass the <code>filter</code> condition. The
   * resulting forest contains only the rows that pass the condition (all of them).</p>
   *
   * <p>The topology of the resulting forest may differ from the original forest - that is, a row may
   * have a different parent in the resulting forest. This happens when a row that has sub-rows is filtered
   * out - in that case, its sub-tree is substituted instead of the parent row.
   * This is different from {@link #filterSoft} method, which preserves the topology.</p>
   *
   * <p>This forest is not modified by this method. If all rows pass the condition, then this forest
   * is returned as the result. If filtering has taken place, a new forest is returned.</p>
   *
   * <p>The filter method is called once for every row in the forest, and a truthy result
   * (as defined in {@link La#accepts}) indicates that the row passes the filter.</p>
   *
   * @param filter filter that returns a truthy value if the row with given ID is allowed to be present in the resulting forest.
   *  Null means "no filtering" - this forest is returned
   * @return a filtered forest (or this forest if all rows satisfy the filter)
   * @see #filterSoft
   */
  @NotNull
  Forest filter(@Nullable La<Long, ?> filter);

  /**
   * <p>Filters this forest by excluding rows that do not pass the <code>filter</code> condition. All rows
   * that contain sub-rows that have passed the filter are also preserved. The
   * resulting forest contains sub-sequence of the original forest with rows having the same parents and depths.</p>
   *
   * <p>Unlike {@link #filter} method, this method preserves the topology of the original forest -
   * all rows in the resulting forest have the same root path as they do in the original forest.</p>
   *
   * <p>This forest is not modified by this method. If all rows pass the condition, then this forest
   * is returned as the result. If filtering has taken place, a new forest is returned.</p>
   *
   * <p>The filter method is called once for every row in the forest, and a truthy result
   * (as defined in {@link La#accepts}) indicates that the row passes the filter.</p>
   * 
   * <p>Note: if you need to filter by hierarchy-based or JQL constraints, see {@link StructureQuery}.</p>
   *
   * @param filter filter that returns a truthy value if a row with given ID should be present in the resulting forest
   *  Null means "no filtering" - this forest is returned
   * @return a filtered forest (or this forest if all rows satisfy the filter)
   */
  @NotNull
  Forest filterSoft(@Nullable La<Long, ?> filter);

  /**
   * <p>Filters this forest by excluding rows that do not pass the <code>filter</code> condition. 
   * All sub-rows of rows that have not passed the filter are also removed.
   * The resulting forest contains only rows that pass the condition, but possibly not all of them.</p>
   *
   * <p>Unlike {@link #filter} method, this method preserves the topology of the original forest -
   * all rows in the resulting forest have the same root path as they do in the original forest. 
   * However, unlike {@link #filterSoft} it achieves that by not including those matching rows  
   * that would change the hierarchy.</p>
   *
   * <p>This forest is not modified by this method. If all rows pass the condition, then this forest
   * is returned as the result. If filtering has taken place, a new forest is returned.</p>
   *
   * <p>The filter method is called once for every row in the forest, and a truthy result
   * (as defined in {@link La#accepts}) indicates that the row passes the filter.</p>
   *
   * @param filter filter that returns a truthy value if a row with given ID should be present in the resulting forest
   *  Null means "no filtering" - this forest is returned
   * @return a filtered forest (or this forest if all rows satisfy the filter)
   */
  @NotNull
  Forest filterHardest(@Nullable La<Long, ?> filter);

  /**
   * Creates a forest that contains the specified row and all its sub-rows from this forest.
   *
   * @param row the root of the sub-tree
   * @return a new forest that contains a copy of the sub-tree rooted at <code>row</code>, 
   *   or an empty forest if the row is not in this forest
   */
  @NotNull
  Forest subtree(long row);

  /**
   * Creates a forest that contains the specified row and all its sub-rows from this forest.
   *
   * @param index index of the root of the sub-tree
   * @return a new forest that contains a copy of the sub-tree rooted at row at <code>index</code>, 
   *   or an empty forest if the index is negative
   */
  @NotNull
  Forest subtreeAtIndex(int index);

  // todo why? there are has already #subtree
  // todo NotNull - only one usage needs the difference between empty forest and null
  @Nullable
  ArrayForest copySubforest(long row);

  @Nullable
  ArrayForest copySubforestAtIndex(int index);

  /**
   * Creates an exact copy of this forest.
   *
   * @return a copy of this forest
   */
  ArrayForest copy();

  // ======================== UTILITIES ========================

  /**
   * <p>This method is used to efficiently traverse all pairs (parent, children) from the end of the forest upwards.</p>
   *
   * <p>This method goes over the forest in the backwards direction and reports to the visitor
   * pairs of (parent, direct children).</p>
   * <p>Invariants:</p>
   * <ul>
   * <li>The number of calls to the visitor is equal to the forest size: every row is reported as a parent once.</li>
   * <li>A child row is reported (as the parent of its own sub-rows) before parent is reported:
   * the iteration goes upwards.</li>
   * <li>A leaf row is reported as a parent with no children (empty children list).</li>
   * </ul>
   * <p>If the forest is modified during iteration, the results are undefined.</p>
   *
   * @param visitor an instance that receives the pairs of parent and children array
   */
  void visitParentChildrenUpwards(ForestParentChildrenVisitor visitor);

  /**
   * <p>This is a more generic version of {@link #visitParentChildrenUpwards} that allows you to effectively
   * process the forest bottom-up, probably saving on memory allocation and search speed.</p>
   *
   * <p>
   * The method goes over the forest in the backwards direction and calls {@code closure} methods for each
   * row:
   * </p>
   * <ul>
   * <li>{@link ForestParentChildrenClosure#visitRow} is called for every row (in the same way {@link #visitParentChildrenUpwards})
   * works;</li>
   * <li>{@code visitRow()} can return a result of the processing;</li>
   * <li>{@link ForestParentChildrenClosure#combine} is called after each call to {@code visitRow()} to aggregate the results
   * of children under the same parent.</li>
   * </ul>
   *
   * @param closure the closure
   * @param <T> the type of the result of processing one row
   * @param <C> the type of the result of processing a number of sub-rows under the same parent
   * @return the result of processing top-level rows in the forest
   * @see #visitParentChildrenUpwards
   * @see ForestParentChildrenClosure
   */
  @Nullable
  <T, C> C foldUpwards(ForestParentChildrenClosure<T, C> closure);

  /**
   * Iterates through each row from top to the bottom. {@code ForestScanner} receives
   * {@link ForestScanControl}, which can be used to cancel the scan, access parents, or skip subtrees.
   *
   * @param scanner the iteratee
   */
  void scanDownwards(ForestScanner scanner);

  /**
   * Utility method for debugging - returns full string representation of the forest, that contains all the information,
   * unlike <code>toString()</code> method, which may be truncated to some character number limit.
   *
   * @return a full string containing all information about this forest
   */
  @NotNull
  String toFullString();
}
