package com.almworks.jira.structure.api.view;

import com.almworks.jira.structure.api.permissions.PermissionRule;
import com.almworks.jira.structure.api.permissions.PermissionSubject;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.PublicApi;
import org.codehaus.jackson.annotate.JsonIgnore;
import org.codehaus.jackson.map.annotate.JsonSerialize;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import javax.xml.bind.annotation.*;
import java.util.Collection;
import java.util.List;

/**
 * <p>Immutable representation of the {@link StructureView} properties. You don't normally interact with
 * this class, but you may find {@link StructureViewBean.Builder} useful for collecting changes
 * to {@code StructureView} properties.</p>
 *
 * <p>Because {@code StructureViewBean} is immutable, it is thread-safe.</p>
 *
 * @see StructureView
 * @see StructureViewBean.Builder
 * @author Igor Sereda
 */
@PublicApi
public class StructureViewBean {
  public static final StructureViewBean EMPTY = new StructureViewBean(0L, "", "", null, null, ViewSpecification.EMPTY);

  private final long myId;
  private final String myName;
  private final String myDescription;
  private final PermissionSubject myOwner;
  private final List<PermissionRule> myPermissions;
  private final ViewSpecification mySpecification;

  private StructureViewBean(Long id, String name, String description, PermissionSubject owner,
    List<PermissionRule> permissions, ViewSpecification specification)
  {
    myId = id == null ? 0 : id;
    myName = name == null ? "" : name;
    myDescription = description == null ? "" : description;
    myOwner = owner;
    myPermissions = permissions;
    mySpecification = specification == null ? ViewSpecification.EMPTY : specification;
  }

  public long getId() {
    return myId;
  }

  @NotNull
  public String getName() {
    return myName;
  }

  @NotNull
  public String getDescription() {
    return myDescription;
  }

  public PermissionSubject getOwner() {
    return PermissionSubject.clone(myOwner);
  }

  public List<PermissionRule> getPermissions() {
    return StructureUtil.copyPermissions(myPermissions);
  }

  @NotNull
  public ViewSpecification getSpecification() {
    return mySpecification;
  }

  public String toString() {
    return "StructureViewBean{" +
      "id=" + myId +
      ", name='" + myName + '\'' +
      ", description='" + myDescription + '\'' +
      ", owner=" + myOwner +
      ", permissions=" + myPermissions +
      ", specification=" + mySpecification +
      '}';
  }

  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;

    StructureViewBean that = (StructureViewBean) o;

    if (myId != that.myId) return false;
    if (!myDescription.equals(that.myDescription)) return false;
    if (!myName.equals(that.myName)) return false;
    if (myOwner != null ? !myOwner.equals(that.myOwner) : that.myOwner != null) return false;
    if (myPermissions != null ? !myPermissions.equals(that.myPermissions) : that.myPermissions != null) return false;
    if (!mySpecification.equals(that.mySpecification)) return false;

    return true;
  }

  public int hashCode() {
    int result = (int) (myId ^ (myId >>> 32));
    result = 31 * result + myName.hashCode();
    result = 31 * result + myDescription.hashCode();
    result = 31 * result + (myOwner != null ? myOwner.hashCode() : 0);
    result = 31 * result + (myPermissions != null ? myPermissions.hashCode() : 0);
    result = 31 * result + mySpecification.hashCode();
    return result;
  }


  /**
   * <p>A builder class for {@link StructureViewBean}, this class can be also used as the input
   * for {@link StructureView#update}.</p>
   *
   * <p>This class is not thread-safe.</p>
   */
  @XmlRootElement(name = "structure-view")
  @XmlType(name = "structure-view", propOrder = {"id", "name", "description", "owner", "permissions", "specification"})
  @JsonSerialize(include = JsonSerialize.Inclusion.NON_NULL)
  public static class Builder implements Cloneable {
    private Long myId;
    private String myName;
    private String myDescription;
    private PermissionSubject myOwner;
    private List<PermissionRule> myPermissions;
    private ViewSpecification.Builder mySpecification;

    /**
     * Creates an empty builder.
     */
    public Builder() {
    }

    /**
     * Creates a builder that is initialized with all properties from the specified bean.
     */
    public Builder(@Nullable StructureViewBean bean) {
      if (bean != null) {
        myId = bean.getId();
        myName = bean.getName();
        myDescription = bean.getDescription();
        myOwner = bean.getOwner();
        myPermissions = bean.getPermissions();
        mySpecification = new ViewSpecification.Builder(bean.getSpecification());
      }
    }

    @SuppressWarnings("CloneDoesntDeclareCloneNotSupportedException")
    public Builder clone() {
      try {
        Builder r = (Builder) super.clone();
        r.myOwner = PermissionSubject.clone(r.myOwner);
        r.myPermissions = StructureUtil.copyPermissionsOrNull(r.myPermissions);
        r.mySpecification = r.mySpecification == null ? null : r.mySpecification.clone();
        return r;
      } catch (CloneNotSupportedException e) {
        throw new AssertionError(e);
      }
    }

    /**
     * Updates properties of the target builder. Each non-null property in this bean is
     * written into target.
     *
     * @param target target bean
     */
    public void updatePropertiesOf(StructureViewBean.Builder target) {
      if (myId != null) target.setId(myId);
      if (myName != null) target.setName(myName);
      if (myDescription != null) target.setDescription(myDescription);
      if (myOwner != null) target.setOwner(myOwner);
      if (myPermissions != null) target.setPermissions(myPermissions);
      if (mySpecification != null) target.setSpecification(mySpecification);
    }

    /**
     * @return the ID of the view or {@code null} if not set
     */
    @XmlAttribute
    public Long getId() {
      return myId;
    }

    /**
     * Sets the ID of the view
     *
     * @param id the ID or {@code null} to clear the property
     */
    public void setId(Long id) {
      myId = id;
    }

    /**
     * @return the name of the view or {@code null} if not set
     */
    @XmlElement
    public String getName() {
      return myName;
    }

    /**
     * Sets the name of the view
     *
     * @param name the name of the view or {@code null} to clear the property
     */
    public void setName(String name) {
      myName = name;
    }

    /**
     * @return the description of the view or {@code null} if not set
     */
    @XmlElement
    public String getDescription() {
      return myDescription;
    }

    /**
     * Sets the description of the view.
     *
     * @param description the description of the view or {@code null} to clear the property
     */
    public void setDescription(String description) {
      myDescription = description;
    }

    /**
     * @return the owner of the view, or {@code null} if not set or if no owner is specified
     */
    @XmlElement
    public PermissionSubject getOwner() {
      return PermissionSubject.clone(myOwner);
    }

    /**
     * @return true if an owner is specified in this builder
     */
    public boolean hasOwner() {
      return myOwner != null;
    }

    /**
     * Sets the owner of this view.
     *
     * @param owner owner or {@code null} to unset.
     */
    public void setOwner(PermissionSubject owner) {
      myOwner = PermissionSubject.clone(owner);
    }

    /**
     * @return permission rules for the view, or empty list if not set
     */
    @NotNull
    @XmlElementRef
    @XmlElementWrapper(name = "permissions")
    public List<PermissionRule> getPermissions() {
      return StructureUtil.copyPermissions(myPermissions);
    }

    /**
     * @return true if permissions are defined in this builder
     */
    public boolean hasPermissions() {
      return myPermissions != null;
    }

    /**
     * Sets the permission rules.
     *
     * @param permissions the list of permission rules for the view, or {@code null} to clear the property
     */
    public void setPermissions(@Nullable Collection<? extends PermissionRule> permissions) {
      myPermissions = StructureUtil.copyPermissionsOrNull(permissions);
    }

    /**
     * @return view specification builder, which can be modified
     */
    @XmlAttribute
    public ViewSpecification.Builder getSpecification() {
      return mySpecification;
    }

    /**
     * Updates view specification builder
     *
     * @param specification new view specification builder (it may be updated after it is installed into this builder)
     */
    public void setSpecification(ViewSpecification.Builder specification) {
      mySpecification = specification;
    }

    /**
     * @return true if the builder contains valid data for the view: that is, if it has a name, a specification and
     * a valid ID.
     */
    @JsonIgnore
    public boolean isValid() {
      return myId != null && myId != 0L && isValidExceptId();
    }

    /**
     * @return true if the builder contains valid data that is needed to create a view. Same as
     * {@link #isValid()} but ID may be missing.
     */
    @JsonIgnore
    public boolean isValidExceptId() {
      return myName != null && myName.trim().length() > 0 && mySpecification != null;
    }

    /**
     * Creates an immutable {@link StructureViewBean} from the state in this builder.
     *
     * @return a new {@code StructureViewBean}
     * @throws IllegalStateException if the current state is not valid
     */
    @NotNull
    public StructureViewBean build() throws IllegalStateException {
      if (!isValid()) {
        throw new IllegalStateException("builder is not in valid state: " + this);
      }
      PermissionSubject owner = PermissionSubject.clone(myOwner);
      List<PermissionRule> permissions = StructureUtil.copyPermissions(myPermissions);
      ViewSpecification spec = mySpecification.build();
      return new StructureViewBean(myId, myName, myDescription, owner, permissions, spec);
    }

    public String toString() {
      return "StructureViewBean.Builder{" +
        "id=" + myId +
        ", name='" + myName + '\'' +
        ", description='" + myDescription + '\'' +
        ", owner=" + myOwner +
        ", permissions=" + myPermissions +
        ", specification=" + mySpecification +
        '}';
    }
  }
}
