package com.almworks.jira.structure.api.effect;

import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;

import java.util.List;

/**
 * An effect provider is responsible for converting {@link StoredEffect effect
 * descriptions} to pieces of {@link Effect runnable code}.
 */
@PublicSpi
public interface EffectProvider {
  /**
   * <p>Converts an effect description to an actual effect implementation,
   * if possible.
   *
   * <p>This method performs all the checks required to make sure that the
   * effect succeeds, e.g. it checks that all referenced entities do exist,
   * and that the current user has sufficient permissions to change them.
   *
   * <p>If the change cannot be performed, returns an error response explaining
   * the problem.
   */
  @NotNull
  EffectResponse resolve(@NotNull StoredEffect effect);

  /**
   * <p>Attempts to reduce effect application time by optimizing the given effects collection. Optimization examples are removing duplicates and
   * resolving conflicts. A conflict is a situation when an effect overrides a previous effect. Conflicts and duplicates may occur when
   * there are duplicate items in the forest or several effectors produce similar effects. The effect provider is not required to optimize all the
   * situations but may attempt to speed up effect application by removing effects or joining several effects into one. When resolving a conflict
   * between several effects the last one should be considered dominant. No other restrictions are imposed here. The method may add new effects
   * and warnings, and nevertheless the results of its work are considered an optimization.
   *
   * <p>The method may be called after all effectors in the effector process have generated effects. There is no guarantee this method will be
   * called at all, because there are situations that clearly do not require optimization.
   *
   * <p>The default implementation returns the source list unchanged (no optimization).
   *
   * @param effects list of effects belonging (by module key) to this EffectProvider, in creation order
   * @return optimized list of effects, not necessarily belonging to this EffectProvider
   */
  @NotNull
  default List<StoredEffect> optimize(@NotNull List<StoredEffect> effects) {
    return effects;
  }
}
