package com.almworks.jira.structure.api.extension.license;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;
import java.util.Objects;

import static com.almworks.jira.structure.api.extension.license.StructureLicenseHosting.LEGACY;
import static com.almworks.jira.structure.api.extension.license.StructureLicenseHosting.SERVER;

public interface LicenseData {
  LicenseData DISABLED = new Disabled();

  String getLicensee();

  StructureLicenseType getLicenseType();

  @Nullable
  String getSerial();

  long getSerialNumber();

  /**
   * @return expiration date, <= 0 if no expiration
   */
  long getExpirationTimestamp();

  /**
   * @return maintenance expiration date, <= 0 if no expiration
   */
  long getMaintenanceExpirationDate();

  /**
   * @return collection of licensed server IDs, or null if not limited
   */
  Collection<String> getLicensedServers();

  boolean isServerLocked();

  int getMaximumNumberOfUsers();

  boolean isUnlimitedNumberOfUsers();

  void extraChecks() throws LicenseException;

  boolean hasExpiration();

  boolean hasMaintenanceExpiration();

  boolean isMarketplaceLicense();

  boolean isValid();

  @NotNull
  StructureLicenseError getLicenseError();

  default boolean isDataCenterCompatible() {
    return getLicenseHosting() != SERVER;
  }

  @NotNull
  default StructureLicenseHosting getLicenseHosting() {
    return LEGACY; // todo: remove default impl?
  }

  @NotNull
  default LicenseSource getLicenseSource() {
    return isMarketplaceLicense() ? LicenseSource.MARKETPLACE : LicenseSource.VENDOR;
  }

  class Disabled implements LicenseData {
    private final StructureLicenseError myError;

    public Disabled() {
      myError = StructureLicenseError.NO_ERROR;
    }

    public Disabled(StructureLicenseError error) {
      myError = error;
    }

    @Override
    public String getLicensee() {
      return "--";
    }

    @Override
    public StructureLicenseType getLicenseType() {
      return StructureLicenseType.DISABLED;
    }

    @Nullable
    @Override
    public String getSerial() {
      return null;
    }

    @Override
    public long getSerialNumber() {
      return 0;
    }

    @Override
    public long getExpirationTimestamp() {
      return 0;
    }

    @Override
    public long getMaintenanceExpirationDate() {
      return 0;
    }

    @Override
    public Collection<String> getLicensedServers() {
      return null;
    }

    @Override
    public boolean isServerLocked() {
      return false;
    }

    @Override
    public int getMaximumNumberOfUsers() {
      return 0;
    }

    @Override
    public boolean isUnlimitedNumberOfUsers() {
      return false;
    }

    @Override
    public void extraChecks() throws LicenseException {
    }

    @Override
    public boolean hasExpiration() {
      return false;
    }

    @Override
    public boolean hasMaintenanceExpiration() {
      return false;
    }

    @Override
    public boolean isMarketplaceLicense() {
      return false;
    }

    @Override
    public boolean isValid() {
      return false;
    }

    @NotNull
    @Override
    public StructureLicenseError getLicenseError() {
      return myError;
    }

    @Override
    @NotNull
    public LicenseSource getLicenseSource() {
      return LicenseSource.NONE;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o) return true;
      if (o == null || getClass() != o.getClass()) return false;
      Disabled disabled = (Disabled) o;
      return myError == disabled.myError;
    }

    @Override
    public int hashCode() {
      return Objects.hash(myError);
    }
  }
}
