package com.almworks.jira.structure.api.sync;

import com.almworks.integers.LongList;
import com.almworks.jira.structure.api.event.JiraChangeType;
import com.almworks.jira.structure.api.structure.StructureListener;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;

import javax.annotation.concurrent.NotThreadSafe;
import java.util.List;

/**
 * <p>This is a temporary object that's passed to a synchronizer and tells what's been changed
 * in structure and in JIRA or in the external system.</p>
 *
 * <p>This data is used in the incremental synchronization to limit what issues should
 * be checked and in what direction the changes should be applied.</p>
 * 
 * <p>Objects of these class are confined to the synchronization thread.</p>
 *
 * @author Igor Sereda
 */
@PublicApi
@NotThreadSafe
public interface IncrementalSyncData {
  /**
   * <p>Returns a detailed list of events that happened in JIRA, in the order of happening.</p>
   * 
   * <p>Note: in JIRA Data Center, the order is approximate and is based on timestamp, so:</p>
   * <ul>
   *   <li>if clocks on different nodes are not synchronized with ms precision, the order is skewed;
   *   <li>if two events happened on different nodes during the same millisecond, their order is undefined.
   * </ul>
   * 
   * @return a list of JIRA events (issue created, updated, issues linked, etc.). Not null.
   */
  @NotNull
  List<SyncEvent.Jira> getJiraEvents();

  /**
   * <p>Returns a detailed list of events that happened in the synchronized structure, in the order of happening.</p>
   *
   * <p>Note: the order is approximate, with the same contract as for {@link StructureListener}.</p>
   * 
   * @return a list of Structure events (rows added/moved/deleted - including both non-issue and issue rows). Not null.
   * */
  @NotNull
  List<SyncEvent.Structure> getStructureEvents();

  /**
   * Returns a sorted unique list of IDs of issues that have been changed in JIRA (edited, linked, etc.)
   * 
   * @param eventTypes if not empty, include only those issues for which there is the event of the corresponding type
   */
  @NotNull
  LongList getJiraChangedIssuesSorted(JiraChangeType... eventTypes);

  /**
   * Returns sorted unique list of IDs of rows that have been changed in structure:
   * <ul>
   *   <li>all rows that were added or copied in the structure,
   *   <li>all moved rows (note: if a row is moved, its subtree is also moved, but it is not reported here),</li>
   *   <li>all deleted rows (note: if a row is removed, its whole subtree is also removed, and it is reported here).
   * </ul>
   * Note that unlike {@link #getStructureChangedIssuesSorted()}, non-issue rows are also reported in this method.
   * This method may work faster than {@link #getStructureChangedIssuesSorted()}.
   *
   * @return rows that were the object of changes in the structure
   */
  @NotNull
  LongList getStructureChangedRowsSorted();

  /**
   * Returns sorted list of IDs of issues that have been changed in structure:
   * <ul>
   *   <li>all issues that were added or copied in the structure,
   *   <li>all moved issues (note: if an issue is moved, its subtree is also moved, but it is not reported here),</li>
   *   <li>all deleted issues (note: if an issue is removed, its whole subtree is also removed, and it is reported here).
   * </ul>
   */
  @NotNull
  LongList getStructureChangedIssuesSorted();

  /**
   * <p>Returns sorted list of row IDs of all parents of the rows affected by this change in Structure
   * (or those used to be parents before the change).</p>
   * 
   * <p>Note that it's not possible to otherwise reach those rows now from the affected rows
   * because the forest has changed.
   * And these rows may be subject of changes themselves and may be not present in the forest.</p>
   *
   * <p>Note that this does not include parents of rows for issues affected by changes in JIRA,
   * such as Delete Issue, Move Issue, etc.</p>
   *
   * @return rows that used to be parent rows of the rows returned by {@link #getStructureChangedRowsSorted()} ()}.
   */
  @NotNull
  LongList getPreviousParentRowsSorted();
}
