package com.almworks.jira.structure.api.attribute.loader.reduce;

import com.almworks.jira.structure.api.attribute.*;
import com.almworks.jira.structure.api.attribute.loader.AggregateAttributeLoader;
import com.almworks.jira.structure.api.attribute.loader.basic.AbstractAggregateLoader;
import com.google.common.collect.Lists;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import java.util.function.Supplier;

import static com.almworks.jira.structure.api.attribute.SharedAttributeSpecs.Param.*;

/**
 * <p>Describes a part of subtree to be used for calculating aggregate values. Subtree type can be:
 * <ul>
 * <li>subtree: full subtree (default),</li>
 * <li>strict: subtree without root node,</li>
 * <li>children: direct children only,</li>
 * <li>leaves: leaves only.</li>
 * </ul>
 * </p>
 *
 * <p>The way it works is that {@link #apply} is called from {@link AggregateAttributeLoader#loadValue AttributeLoader.Aggregate.loadValue},
 * and it passes the values corresponding to the specific part of the subtree to a {@link ValueReducer}.
 * </p>
 *
 * <p>
 * Implementations provide the way to effectively calculate reduction on given subtree.
 * They may use (but may ignore) {@link AttributeValue#getValue() AttributeValue.value} of the current row.
 * They may also use {@link AttributeValue#getLoaderData(Class) AttributeValue.loaderData} for intermediate calculations.
 * Every implementation defines its own contract for attribute value and loader data.
 * If an implementation uses loader data, it should check for errors.
 * It doesn't need to check for errors in attribute values because the check should be performed by the caller code.
 * </p>
 *
 * <p>
 * This class is intended to be an inner utility of {@link ReducingAggregateLoader} and similar classes.
 * First an {@link ReductionStrategy} instance is retrieved using {@link ReductionStrategy#forAttributeSpec},
 * which looks at {@link SharedAttributeSpecs.Param#TYPE} parameter in the attribute specification.
 * Default recognized subtree types are described in {@link ReductionStrategy#forStrategyType}.
 * Typical usage is to implement aggregation modifiers in formulas.
 * </p>
 *
 * @param <T> type of reduced data
 * @see ValueReducer
 * @see ReducingAggregateLoader
 */
public interface ReductionStrategy<T> {

  /**
   * Reduce value of the current node and children values to the single value using the specified {@code ValueReducer}.
   * This method is designed to be used from {@code AttributeLoader.Aggregate}.
   * Reduction can be performed over any subtree and it's assumed that every {@code ReductionStrategy}
   * describes in this method a reduction for a single known subtree type.
   *
   * @param selfSupplier supplier for value of current row. Supplier can be or be not called from this method depending on type of subtree
   * this strategy describes. It's assumed that calling supplier can be heavy operation so it shouldn't be called more than once for each row from
   * this method.
   * @param children values of direct children rows supplied by {@code AttributeLoader.Aggregate.loadValue}
   * @param reducer instance that is responsible for reducing but can ignore actual form of handled subtree
   * @return reduced subtree value
   */
  @NotNull
  AttributeValue<T> apply(@NotNull Supplier<AttributeValue<T>> selfSupplier, @NotNull List<AttributeValue<T>> children,
    @NotNull ValueReducer<T> reducer);

  /**
   * Look for known strategy type from the spec and return default ({@code SUBTREE}) value if not found.
   *
   * @param attributeSpec spec to search for strategy type
   * @return strategy type
   */
  @NotNull
  static String getStrategyType(@NotNull AttributeSpec<?> attributeSpec) {
    return attributeSpec.getParamsMap().getOrDefault(TYPE, SUBTREE).toString();
  }

  /**
   * Look for strategy type from the spec and try to return default strategy for known types.
   *
   * @param attributeSpec spec to search for strategy type
   * @return reduction strategy
   * @throws IllegalArgumentException iff strategy type is unknown
   */
  @NotNull
  static <T> ReductionStrategy<T> forAttributeSpec(@NotNull AttributeSpec<?> attributeSpec) {
    String strategyType = getStrategyType(attributeSpec);
    return forStrategyType(strategyType);
  }

  /**
   * Try to return default strategy for known types.
   *
   * @param strategyType strategy type
   * @return reduction strategy
   * @throws IllegalArgumentException iff strategy type is unknown
   */
  @SuppressWarnings("unchecked")
  @NotNull
  static <T> ReductionStrategy<T> forStrategyType(@NotNull String strategyType) {
    switch (strategyType) {
    case CHILDREN:
      return (ReductionStrategy<T>) ChildrenReductionStrategy.INSTANCE;
    case LEAVES:
      return (ReductionStrategy<T>) LeavesReductionStrategy.INSTANCE;
    case STRICT:
      return (ReductionStrategy<T>) StrictReductionStrategy.INSTANCE;
    case SUBTREE:
      return (SubtreeReductionStrategy<T>) SubtreeReductionStrategy.INSTANCE;
    default:
      throw new IllegalArgumentException("unknown strategy type '" + strategyType + "'");
    }
  }


  abstract class AbstractReductionStrategy<T> implements ReductionStrategy<T> {
    static final Logger logger = LoggerFactory.getLogger(ReductionStrategy.class);

    /**
     * Calls a function that processes self value and takes care of the case where self value is an error.
     *
     * @param selfSupplier provides value for the row
     * @param errorUnawareProcessor the processor of self value - it may or may not call the supplier for the self value
     * @return the result of calling {@code errorUnawareProcessor} or an error attribute value in case self supplier was called and returned an error
     */
    @NotNull
    AttributeValue<T> processSelfValue(Supplier<AttributeValue<T>> selfSupplier, Function<Supplier<T>, T> errorUnawareProcessor) {
      AtomicReference<AttributeValue<T>> error = new AtomicReference<>();
      Supplier<T> errorRecordingSupplier = () -> {
        AttributeValue<T> value = selfSupplier.get();
        if (value == null) return null;
        if (value.isError()) error.set(value);
        return value.getValue();
      };
      T value = errorUnawareProcessor.apply(errorRecordingSupplier);

      return error.get() != null ? error.get() : AttributeValue.ofNullable(value);
    }

    /**
     * Note: if we're using additional data in AttributeValue then we require that each calculated value has this data.
     */
    @NotNull
    AttributeValue<T> reduceValuesInChildrenAdditionalData(@NotNull List<AttributeValue<T>> children, @NotNull ValueReducer<T> reducer) {
      List<AttributeValue<T>> additionalDataList = Lists.transform(children, this::getAdditionalData);
      AttributeValue<T> error = AbstractAggregateLoader.firstChildError(additionalDataList);
      if (error != null) return error;
      try {
        T value = reducer.reduce(Lists.transform(additionalDataList, v -> v != null ? v.getValue() : null));
        return AttributeValue.ofNullable(value);
      } catch (ClassCastException e) {
        // CCE can be fired if a child's value has additional data of a different type
        logger.error("error reducing children values", e);
        return AttributeValue.error();
      }
    }

    @NotNull
    @SuppressWarnings("unchecked")
    <D> AttributeValue<D> getAdditionalData(AttributeValue<D> child) {
      if (child == null) return AttributeValue.undefined();
      AttributeValue<D> value = child.getLoaderData(AttributeValue.class);
      return value != null ? value : AttributeValue.undefined();
    }

    /**
     * The list can contain null.
     */
    @NotNull
    List<T> valueList(@NotNull List<AttributeValue<T>> attributeValueList) {
      return Lists.transform(attributeValueList, v -> v == null ? null : v.getValue());
    }
  }


  /**
   * <p>Main value: the aggregation based only on children.</p>
   * <p>Additional data: {@code AttributeValue<T>} that contains only self value of the row.</p>
   * <p>Calculation idea: sum the additional values from all children.</p>
   */
  class ChildrenReductionStrategy<T> extends AbstractReductionStrategy<T> {
    private static final ChildrenReductionStrategy<?> INSTANCE = new ChildrenReductionStrategy<>();

    @NotNull
    @Override
    public AttributeValue<T> apply(@NotNull Supplier<AttributeValue<T>> selfSupplier, @NotNull List<AttributeValue<T>> children,
      @NotNull ValueReducer<T> reducer)
    {
      AttributeValue<T> value = reduceValuesInChildrenAdditionalData(children, reducer);
      return value.withData(selfSupplier.get());
    }
  }


  /**
   * <p>Main value: the aggregation based only on leaves.</p>
   * <p>Additional data: none.</p>
   * <p>Calculation idea: aggregate only values from children.</p>
   */
  class LeavesReductionStrategy<T> extends AbstractReductionStrategy<T> {
    private static final LeavesReductionStrategy<?> INSTANCE = new LeavesReductionStrategy<>();

    @NotNull
    @Override
    public AttributeValue<T> apply(@NotNull Supplier<AttributeValue<T>> selfSupplier, @NotNull List<AttributeValue<T>> children,
      @NotNull ValueReducer<T> reducer)
    {
      if (children.isEmpty()) {
        return processSelfValue(selfSupplier, reducer::convert);
      } else {
        return AttributeValue.ofNullable(reducer.reduce(valueList(children)));
      }
    }
  }

  /**
   * <p>Main value: the aggregation based on the whole subtree without self row.</p>
   * <p>Additional data: value that does include self.</p>
   * <p>Calculation idea: sum the additional values from all children.</p>
   */
  class StrictReductionStrategy<T> extends AbstractReductionStrategy<T> {
    private static final StrictReductionStrategy<?> INSTANCE = new StrictReductionStrategy<>();

    @NotNull
    @Override
    public AttributeValue<T> apply(@NotNull Supplier<AttributeValue<T>> selfSupplier, @NotNull List<AttributeValue<T>> children,
      @NotNull ValueReducer<T> reducer)
    {
      if (children.isEmpty()) {
        AttributeValue<T> attributeValue = AttributeValue.undefined();
        return attributeValue.withData(processSelfValue(selfSupplier, reducer::convert));
      }
      AttributeValue<T> attributeValue = reduceValuesInChildrenAdditionalData(children, reducer);
      AttributeValue<T> withSelf = processSelfValue(selfSupplier, supplier -> reducer.merge(supplier, attributeValue::getValue));
      return attributeValue.withData(withSelf);
    }
  }


  /**
   * <p>Main value: the aggregation based only on all children.</p>
   * <p>Additional data: none.</p>
   */
  class SubtreeReductionStrategy<T> extends AbstractReductionStrategy<T> {
    private static final SubtreeReductionStrategy<?> INSTANCE = new SubtreeReductionStrategy<>();

    @NotNull
    @Override
    public AttributeValue<T> apply(@NotNull Supplier<AttributeValue<T>> selfSupplier, @NotNull List<AttributeValue<T>> children,
      @NotNull ValueReducer<T> reducer)
    {
      return processSelfValue(selfSupplier, supplier -> reducer.merge(supplier, valueList(children)));
    }
  }
}
