package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeSpec;
import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicApi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.concurrent.TimeUnit;

/**
 * <p>This interface adds methods to {@link AttributeContext} that are used only by attribute loaders.</p>
 *
 * <p>Each type of attribute loader will receive a more specific type of context with additional methods
 * based on what that type of loaders might need.</p>
 *
 * @see AttributeContext
 * @see ItemAttributeContext
 * @see SingleRowAttributeContext
 * @see DerivedAttributeContext
 * @see AggregateAttributeContext
 * @see PropagateAttributeContext
 * @see ScanningAttributeContext
 */
@PublicApi
public interface AttributeLoaderContext extends AttributeContext {
  /**
   * <p>Gets the value of dependency attribute for the row being calculated.</p>
   *
   * <p>The loader must declare the dependency in order to get its value here.
   * The attribute system guarantees that the dependency attributes are loaded first.</p>
   *
   * @param dependency attribute spec of the dependency
   * @param <V> type of the value
   * @return value of the dependency, as {@code AttributeValue} wrapper
   * @throws IllegalArgumentException if the dependency attribute was not declared
   * @see AttributeLoader#getAttributeDependencies()
   */
  @NotNull
  <V> AttributeValue<V> getDependencyAttributeValue(@NotNull AttributeSpec<V> dependency);

  /**
   * <p>Gets the value of dependency attribute for the row being calculated.</p>
   *
   * <p>The loader must declare the dependency in order to get its value here.
   * The attribute system guarantees that the dependency attributes are loaded first.</p>
   *
   * @param dependency attribute spec of the dependency
   * @param <V> type of the value
   * @return value of the dependency
   * @throws IllegalArgumentException if the dependency attribute was not declared
   * @see AttributeLoader#getAttributeDependencies()
   */
  @Nullable
  default <V> V getDependencyValue(@NotNull AttributeSpec<V> dependency) {
    return getDependencyAttributeValue(dependency).getValue();
  }

  /**
   * <p>Used to indicate the trail of the value being calculated.</p>
   *
   * <p>A trail is an additional set of items that were used to produce the value for the currently calculated
   * row or item. It is important to indicate the trail because when those additional items change, we need to
   * recalculate this value.</p>
   *
   * <p>For example, if we calculate the display name of the issue's assignee, we use the corresponding
   * "user" object. Even if the issue's assignee does not later change, the user's display name might, so in order
   * to catch that and recalculate the value, we need add the user item as the trail for this value.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#TRAIL} to use this method.</p>
   *
   * @param trail the set of items used to calculate this value
   */
  void addTrail(@Nullable TrailItemSet trail);

  /**
   * <p>Used to indicate the trail of the value being calculated.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#TRAIL} to use this method.</p>
   *
   * @param item a single item that was used to calculate this value
   * @see #addTrail(TrailItemSet)
   */
  default void addTrail(@Nullable ItemIdentity item) { addTrail(TrailItemSet.of(item)); }

  /**
   * <p>Calling this method makes the value being currently calculated expire by timeout. When the value expires, it becomes outdated and will
   * be recalculated the next time it is requested.</p>
   *
   * <p>Any value that depends on one or more expiring values will also have an expiration time equal to the earliest expiration time of its
   * dependencies.</p>
   *
   * <p>Calling this method multiple times will result in the earliest expiration time being used.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#CURRENT_TIME} to use this method.</p>
   *
   * <p>If {@code CURRENT_TIME} context dependency is declared, but {@code valueExpires()} is not called, a default expiration period will
   * be used. It is set to 30 seconds and can be changed via setting {@code structure.attribute.timedValueRefresh} dark feature (in milliseconds).</p>
   *
   * @param ttl amount of time after which the currently calculated value should expire, must be a positive number
   * @param unit time unit
   * @throws IllegalArgumentException if {@code ttl <= 0}
   * @see AttributeLoader
   * @see AttributeLoader#getContextDependencies
   */
  void valueExpires(long ttl, @NotNull TimeUnit unit);
}
