package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.attribute.AttributeValue;
import com.almworks.jira.structure.api.item.ItemIdentity;
import com.atlassian.annotations.PublicSpi;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Collection;

/**
 * <p>Item loaders calculate attribute value for a particular item, based on {@link ItemIdentity}. The value loaded does not depend on
 * a particular forest and generally will be the same for all structures (unless structure context dependency is declared, see below).</p>
 *
 * <p>Values calculated by item loaders are stored in the per-item cache and shared between structures.</p>
 *
 * <p>Item attributes may, but should not depend on row-based attributes. If that happens though, they will be stored in a per-forest cache.
 * If you have such an attribute, consider splitting it into an item and derived attributes.</p>
 *
 * <p>If an item attribute declares context dependency on structure (by including {@link AttributeContextDependency#STRUCTURE} in the
 * result of {@link AttributeLoader#getContextDependencies()}), then, although it will still work as an item attribute, it will be able
 * to return different values for the same item, depending on the current structure (retrieved via {@link AttributeLoaderContext#getBaseStructureId()}).</p>
 *
 * @param <T> type of the loaded value
 *
 * @see AttributeLoader
 */
@PublicSpi
public interface ItemAttributeLoader<T> extends AttributeLoader<T> {
  /**
   * <p>Lets the loader declare if it can handle items of a given type. If the loader returns {@code false} for some type, its {@link #loadValue}
   * function will not be called for items of that type; also, non-supported items will be excluded from the collection passed to {@link #preload} method.</p>
   *
   *  <p>The returned value must be the same for the same item type throughout the lifetime of the object.</p>
   *
   * @param itemType the type of item
   * @return true if the loader supports this item type.
   * @see ItemIdentity
   */
  boolean isItemTypeSupported(@NotNull String itemType);

  /**
   * <p>The loading function. The implementation is expected to take {@code itemId} or the item object (see {@link ItemAttributeContext#getItem})
   * and provide a value.</p>
   *
   * <p>Attribute system's contract:</p>
   * <ul>
   *   <li>Before this method is called, {@link #isItemTypeSupported} method has been called at least once for the type of the
   *   passed item, and this loader has returned {@code true}.</li>
   *   <li>Before this method is called, {@link #preload} has been called for a set of items that included {@code itemId}, and that was
   *   a part of the same loading process, sharing the same context (so all values put into the context are still there).</li>
   * </ul>
   *
   * <p>If the value is missing or empty, the loader should return {@link AttributeValue#undefined()}. If for some reason this loader
   * is not applicable for the given item, it should return {@code null}.</p>
   *
   * @param itemId item to load the value for
   * @param context loading context
   * @return attribute value, or {@code null} to pass on loading a value for this item
   */
  @Nullable
  AttributeValue<T> loadValue(@NotNull ItemIdentity itemId, @NotNull ItemAttributeContext context);

  /**
   * <p>Optional method to perform any bulk actions on a set of items before {@link #loadValue} method is called for each item in the collection.</p>
   *
   * <p>The results of the preloading should be stored using {@link AttributeContext#putObject}.
   * <strong>Do not store the preloading results as loader's instance fields!</strong> All loaders should be stateless.</p>
   *
   * <p>Note that this method may be called several times during a single loading operation, for arbitrary sets of items, including those already
   * pre-loaded. Be careful not to overwrite one execution's result with another's.</p>
   *
   * <p>When this method is called, the attribute system excludes any non-supported items from the list of passed items
   * (checking {@link #isItemTypeSupported}).</p>
   *
   * <p>The loader's dependencies may not be loaded at the time of the call. In fact, there's no way to access the dependency values.</p>
   *
   * @param itemIds a collection of item IDs about to be loaded
   * @param context loading context
   */
  default void preload(@NotNull Collection<ItemIdentity> itemIds, @NotNull AttributeContext context) {}
}
