package com.almworks.jira.structure.api.attribute.loader;

import com.almworks.jira.structure.api.export.*;
import com.almworks.jira.structure.api.forest.ForestSpec;
import com.almworks.jira.structure.api.i18n.I18n;
import com.almworks.jira.structure.api.util.StructureUtil;
import com.atlassian.annotations.PublicApi;
import com.atlassian.jira.user.ApplicationUser;
import com.atlassian.jira.util.I18nHelper;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.util.Locale;
import java.util.TimeZone;

/**
 * <p>An {@code AttributeContext} is associated with every attribute request and Structure export request. The context
 * contains some basic information, like the current user and JIRA base URL, and it can also contain other data
 * associated with the current request by Structure or other plug-ins. The current context is passed into every call
 * to {@link AttributeLoaderProvider AttributeLoaderProvider}, {@link AttributeLoader AttributeLoader},
 * {@link ExportRendererProvider ExportRendererProvider}, and {@link ExportRenderer ExportRenderer}.</p>
 *
 * <p>Please note that the context key space is shared by all apps, so we advise you to use app-specific keys
 * (e.g. singleton objects or custom enums) to avoid conflicts.</p>
 *
 * <p>Note that when attribute loading is performed, the context values, such as current user, require the attribute loader to declare
 * a corresponding context dependency.</p>
 *
 * @see ExportContextKeys
 * @see ExportRequestContext
 * @see ExportRenderContext
 * @see AttributeLoaderContext
 */
@PublicApi
public interface AttributeContext {
  /**
   * <p>Returns the user performing the calculation or export.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#USER} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return the logged-in user, or {@code null} if anonymous
   * @see AttributeLoader#getContextDependencies
   */
  @Nullable
  ApplicationUser getUser();

  /**
   * <p>Provides i18n helper.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#USER_LOCALE} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return the {@link I18nHelper} instance for the current user
   * @see AttributeLoader#getContextDependencies
   */
  @NotNull
  I18nHelper getI18nHelper();

  /**
   * @return The {@link I18n} instance for the current user.
   */
  @NotNull
  I18n getI18n();

  /**
   * <p>Returns the current user's locale.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#USER_LOCALE} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return the current user's {@link Locale}
   */
  @NotNull
  Locale getLocale();

  /**
   * <p>Returns the current user's time zone.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#USER_TIMEZONE} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return the current user's {@link TimeZone}
   */
  @NotNull
  TimeZone getTimeZone();

  /**
   * <p>Returns the ID of the structure for which the attribute is being calculated. Returns 0 if the loading will happen
   * on a query or a structure without an ID.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#STRUCTURE} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return the ID of the structure being loaded, or 0 if there's no permanent structure
   */
  long getBaseStructureId();

  /**
   * <p>Returns the current wall time at the moment when calculation or export has started. This value will be consistently the same
   * for all attributes. It is advised to use this value instead of {@link System#currentTimeMillis()}.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#CURRENT_TIME} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return wall time, fixed at the beginning of the loading process
   */
  long getLoadTimeMillis();

  /**
   * <p>Returns the current system timer value (from {@link System#nanoTime()}) at the moment when calculation or export has started.
   * This value will be consistently the same for all attributes. It is advised to use this value instead of a call to {@code System}.</p>
   *
   * <p>Attribute loaders are required to declare {@link AttributeContextDependency#CURRENT_TIME} to use this method.
   * If called by an {@link AttributeLoaderProvider}, makes the returned attribute loader non-cacheable and also requires that
   * the loader also declares the same dependency.</p>
   *
   * @return system nano time counter, fixed at the beginning of the loading process
   */
  long getLoadTimeNanos();

  /**
   * Utility method, provides the base URL of the current instance.
   *
   * @return base URL of this Jira
   */
  @NotNull
  String getBaseUrl();

  /**
   * <p>Associates an arbitrary {@code value} with the given request. The value can be retrieved by passing the same
   * {@code key} to {@link #getObject(Object)}.</p>
   *
   * <p>Please note that {@code AttributeContext} instances are shared by all attribute loader providers
   * and export renderer providers from all apps. (See {@link AttributeLoaderProvider} and {@link ExportRendererProvider}).
   * Therefore, we recommend you to use app-specific keys (e.g. singleton objects or custom enums) to
   * avoid conflicts.</p>
   *
   * <p>Some system keys are read-only and cannot be changed. Only use the keys that your code owns.</p>
   *
   * @param key value key
   * @param value the value; pass {@code null} to clear the object associated with the given {@code key}
   * @throws IllegalArgumentException if {@code key} is read-only
   */
  void putObject(@NotNull Object key, @Nullable Object value);
  // todo use TypedKey instead of Object

  /**
   * Retrieves the value associated with a key.
   *
   * @param key value key
   * @param <T> the expected type of the value.
   * @return the object associated with the {@code key}, or {@code null} if the object is not in the context
   * @throws ClassCastException if the value is not an instance of type {@code <T>}
   */
  @Nullable
  <T> T getObject(@NotNull Object key);
}
